<?php

namespace Dealerdirect\Agreements\Models;

use Doctrine\DBAL\Connection;
use Dealerdirect\Agreements\Interfaces\Model;

abstract class AbstractModel implements Model
{
    /** @var Connection $dbal */
    private $dbal;
    /** @var int $deliverId */
    private $deliverId;

    /**
     * {@inheritDoc}
     */
    abstract protected function getPurchaseQuery(): string;

    /**
     * {@inheritDoc}
     */
    abstract protected function getSaleQuery(): string;

    /**
     * {@inheritDoc}
     */
    public function __construct(Connection $dbal, int $deliverId)
    {
        $this->dbal = $dbal;
        $this->deliverId = $deliverId;
    }

    /**
     * {@inheritDoc}
     */
    public function getDeliverId(): int
    {
        return $this->deliverId;
    }

    /**
     * {@inheritDoc}
     */
    public function getLocale(): string
    {
        return $this->locale;
    }

    /**
     * {@inheritDoc}
     */
    public function getVehicleType(): string
    {
        return $this->vehicleType;
    }

    /**
     * {@inheritDoc}
     */
    public function getSaleData(): array
    {
        return $this->fetchData(
            $this->getSaleQuery(),
            ['deliverId' => $this->deliverId]
        );
    }

    /**
     * {@inheritDoc}
     */
    public function getPurchaseData(): array
    {
        return $this->fetchData(
            $this->getPurchaseQuery(),
            ['deliverId' => $this->deliverId]
        );
    }

    /**
     * {@inheritDoc}
     */
    public function log(string $action, $user = 0, $ipAddress = null): void
    {
        $this->dbal->insert(
            $this->logTable,
            [
                'leveringref' => $this->deliverId,
                'actie' => $action,
                'gebruiker' => $user,
                'ip' => $ipAddress ?? $_SERVER['REMOTE_ADDR']
            ]
        );
    }

    /**
     * Fetches data with DBAL
     *
     * @param  string $query
     * @param  array  $params
     * @return array
     * @throws \Exception
     */
    protected function fetchData(string $query, array $params): array
    {
        $data = $this->dbal->fetchAssociative($query, $params);
        if (!is_array($data) || empty($data)) {
            throw new \Exception('Error, no data!');
        }

        return $data;
    }
}
