<?php

namespace Dealerdirect\Agreements;

use DateTime;
use DealerDirect\Generic\Category\Gender;
use DealerDirect\Generic\Category\Locale;
use Dealerdirect\PhoneNumber\PhoneNumber;
use DealerDirect\Generic\Category\VehicleType;
use Dealerdirect\PhoneNumber\Enum\CountryCode;

class Utils
{
    /**
     * Sanitizes phone numbers, with country prefix.
     *
     * @param  integer $countryId
     * @param  string  $phone
     * @return string
     */
    public static function sanitizePhoneNumber(int $countryId, string $phone) : string
    {
        if (!empty($phone)) {
            try {
                CountryCode::assertExists($countryId);
                $phone = PhoneNumber::from($countryId, $phone);
            } catch (\InvalidArgumentException $exception) {
                //Fail silence
            }
        }

        return $phone;
    }

    /**
     * @param string|null $phone1
     * @param string|null $phone2
     * @param int|null $countryId
     *
     * @return string
     */
    public static function combinePhoneNumbers($phone1, $phone2, $countryId = null) : string
    {
        if ($countryId !== null) {
            $phone1 = self::sanitizePhoneNumber($countryId, $phone1);
            $phone2 = self::sanitizePhoneNumber($countryId, $phone2);
        }

        return self::combineValues(' / ', [$phone1, $phone2]);
    }

    /**
     * @param string $separator
     * @param array  $values
     *
     * @return string
     */
    public static function combineValues(string $separator, array $values) : string
    {
        $values = array_unique(array_filter($values));

        return implode($separator, $values);
    }

    /**
     * Translates id [0,1,2] in to [unknown,yes,no]
     *
     * @param  boolean $value
     * @param  string  $locale
     * @return string
     */
    public static function translateUnknownYesNo(int $value, string $locale) : string
    {
        $translations = [
            Locale::NL_NL => [
                0 => 'Onbekend',
                1 => 'Ja',
                2 => 'Nee',
            ],
            Locale::NL_BE => [
                0 => 'Onbekend',
                1 => 'Ja',
                2 => 'Nee',
            ],
            Locale::FR_BE => [
                0 => 'Inconnu',
                1 => 'Oui',
                2 => 'Non',
            ],
            Locale::DE_DE => [
                0 => 'Unbekannt',
                1 => 'Ja',
                2 => 'Nein',
            ],
        ];

        if (isset($translations[$locale][$value])) {
            return $translations[$locale][$value];
        } elseif (isset($translations[Locale::NL_NL][$value])) {
            return $translations[Locale::NL_NL][$value];
        }

        return $value;
    }

    /**
     * @param boolean $value
     * @param string  $locale
     *
     * @return string
     */
    public static function translateYesNo($value, $locale)
    {
        switch ($locale) {
            case Locale::NL_NL: // Fall through
            case Locale::NL_BE: // Fall through
            default:
                $result = $value ? 'ja' : 'nee';
                break;
            case Locale::FR_BE:
                $result = $value ? 'oui' : 'non';
                break;
            case Locale::DE_DE:
                $result = $value ? 'ja' : 'nein';
                break;
        }

        return $result;
    }

    /**
     * Checks if vehicleType is Motor
     *
     * @param  string $vehicleType
     * @return boolean
     */
    public static function isMotor(string $vehicleType) : bool
    {
        return in_array($vehicleType, [VehicleType::MOTOR, VehicleType::SCOOTER]);
    }

    /**
     * Checks if vehicleType is Car.
     *
     * @param  string $vehicleType
     * @return boolean
     */
    public static function isCar(string $vehicleType) : bool
    {
        return !self::isMotor($vehicleType);
    }

    /**
     * Gets the locale by company pri_lang(db field)
     *
     * @param  integer $lang
     * @return string
     */
    public static function getLocaleByCompanyLang(int $lang) : string
    {
        $locales = [
            1 => Locale::NL_NL,
            2 => Locale::NL_BE,
            3 => Locale::DE_DE,
            4 => Locale::FR_BE,
        ];

        return $locales[$lang] ?? Locale::NL_NL;
    }

    /**
     * @param  integer $reference
     * @return array
     */
    public static function getSiteDataByReference(int $reference, string $vehicleType) : array
    {
        $default = self::isMotor($vehicleType) ? 4 : 1;
        $siteData = [
            1 => [
                'locale' => Locale::NL_NL,
                'vehicleType' => VehicleType::CAR,
            ],
            2 => [
                'locale' => Locale::NL_BE,
                'vehicleType' => VehicleType::CAR,
            ],
            4 => [
                'locale' => Locale::NL_NL,
                'vehicleType' => VehicleType::MOTOR,
            ],
            40 => [
                'locale' => Locale::NL_BE,
                'vehicleType' => VehicleType::MOTOR,
            ],
            400 => [
                'locale' => Locale::DE_DE,
                'vehicleType' => VehicleType::MOTOR,
            ],
            5 => [
                'locale' => Locale::NL_NL,
                'vehicleType' => VehicleType::SCOOTER,
            ],
        ];

        return $siteData[$reference] ?? $siteData[$default];
    }

    /**
     * Gets the phone number by site reference
     *
     * @param  integer $reference
     * @return string
     */
    public static function getAgentPhoneByReference(int $reference) : string
    {
        $agentPhones = [
            4 => '053 - 20 844 49',
            40 => '016 / 98.03.31',
            400 => '02131 - 7 51 25 28',
            5 => '053 - 208 44 49',
        ];

        return $agentPhones[$reference] ?? $agentPhones[4];
    }

    /**
     * Formats date or return default value
     *
     * @param  string|null $date
     * @param  string      $format
     * @param  string      $default
     * @return string
     */
    public static function formatDateOrDefault($date, string $format = 'd-m-Y', string $default = '-') : string
    {
        return empty($date)
            ? $default
            : (new DateTime($date))->format($format);
    }

    /**
     * Translates gender to salutation
     *
     * @param  string  $locale
     * @param  integer $gender
     * @return string
     */
    public static function getSalutationByGender(string $locale, int $gender) : string
    {
        Locale::assertExists($locale);
        Gender::assertExists($gender);
        switch ($locale) {
            case Locale::DE_DE:
                $result = ($gender === Gender::FEMALE) ? 'Sehr geehrte Frau' : 'Sehr geehrter Herr';
                break;
            case Locale::FR_BE:
            case Locale::FR_FR:
                $result = ($gender === Gender::FEMALE) ? 'Chère madame' : 'Cher monsieur';
                break;
            default:
                $result = '';
                break;
        }

        return $result;
    }
}
