<?php

declare(strict_types=1);

namespace Dealerdirect\Agreements\Tests\Enums;

use Dealerdirect\Agreements\Enums\DealerdirectLanguage;
use Dealerdirect\Agreements\Enums\Locale;
use Dealerdirect\Agreements\Exceptions\InvalidSiteReferenceException;
use Dealerdirect\Agreements\Tests\TestCase;
use Dealerdirect\Generic\Category\Site;

/**
 * @covers \Dealerdirect\Agreements\Enums\Locale
 */
final class LocaleTest extends TestCase
{
    /**
     * @dataProvider provideSiteReference
     */
    public function test_method_from_site_reference(int $siteReference, Locale $expectedLocale): void
    {
        $locale = Locale::fromSiteReference($siteReference);

        $this->assertSame($expectedLocale, $locale);
    }

    /**
     * @covers \Dealerdirect\Agreements\Exceptions\InvalidSiteReferenceException
     */
    public function test_method_from_site_reference_with_unknown_site_reference(): void
    {
        $this->expectException(InvalidSiteReferenceException::class);
        $this->expectExceptionMessage('Unknown site reference: 999');

        Locale::fromSiteReference(999);
    }

    public function test_method_to_dealerdirect_language(): void
    {
        $this->assertSame(DealerdirectLanguage::NL, Locale::NL_NL->toDealerdirectLanguage());
        $this->assertSame(DealerdirectLanguage::NL_BE, Locale::NL_BE->toDealerdirectLanguage());
        $this->assertSame(DealerdirectLanguage::DE, Locale::DE_DE->toDealerdirectLanguage());
        $this->assertSame(DealerdirectLanguage::FR, Locale::FR_BE->toDealerdirectLanguage());
    }

    /**
     * @return array<int,array{0:int,1:Locale}>
     */
    public static function provideSiteReference(): array
    {
        return [
            [Site::REF_BICYCLE_NL, Locale::NL_NL],
            [Site::REF_CAR_NL, Locale::NL_NL],
            [Site::REF_CAR_MARKTPLAATS_NL, Locale::NL_NL],
            [Site::REF_MOTO_NL, Locale::NL_NL],
            [Site::REF_SCOOTER_NL, Locale::NL_NL],
            [Site::REF_CAR_BE_NL, Locale::NL_BE],
            [Site::REF_MOTO_BE, Locale::NL_BE],
            [Site::REF_CAR_DE, Locale::DE_DE],
            [Site::REF_MOTO_DE, Locale::DE_DE],
        ];
    }
}
