<?php

declare(strict_types=1);

namespace Dealerdirect\Agreements\Tests\Pdf;

use Dealerdirect\Agreements\DataObjects\Font;
use Dealerdirect\Agreements\Enums\Locale;
use Dealerdirect\Agreements\Exceptions\PdfGenerationException;
use Dealerdirect\Agreements\Pdf\Generator;
use Dealerdirect\Agreements\Pdf\Templates\AbstractTemplate;
use Dealerdirect\Agreements\Tests\TestCase;
use Dealerdirect\Agreements\Tests\Traits\DataObjectFixtures;
use Dompdf\Dompdf;

/**
 * @covers \Dealerdirect\Agreements\Pdf\Generator
 * @covers \Dealerdirect\Agreements\Pdf\Templates\AbstractTemplate
 * @covers \Dealerdirect\Agreements\Enums\Locale
 */
final class GeneratorTest extends TestCase
{
    use DataObjectFixtures;

    private Generator $generator;

    public function setUp(): void
    {
        parent::setUp();

        $domPdf = new Dompdf();

        $dataObject = $this->getMockForDataObject(['foo' => 'bar']);

        $template = $this->getMockForAbstractClass(
            AbstractTemplate::class,
            [Locale::NL_NL, $dataObject, $this->getMockForTranslator(), $this->getResourcesPath()]
        );
        $template->method('getTemplate')->willReturn('foo-bar.php');
        $template->method('getFonts')->willReturn([
            $this->createMock(Font::class),
        ]);

        $this->generator = new Generator($domPdf, $template);
    }

    public function test_generate_should_return_pdf(): void
    {
        $pdf = $this->generator->generate();

        $this->assertIsString($pdf);
        $this->assertStringStartsWith('%PDF', $pdf);
    }

    public function test_from_template_should_return_new_instance(): void
    {
        $generator = Generator::fromTemplate($this->generator->getTemplate());

        $this->assertInstanceOf(Generator::class, $generator);
    }

    /**
     * @covers \Dealerdirect\Agreements\Exceptions\PdfGenerationException
     */
    public function test_generate_should_throw_exception_when_failed_to_generate_pdf(): void
    {
        $this->expectException(PdfGenerationException::class);

        $domPdf = $this->createPartialMock(Dompdf::class, ['output', 'render', 'getFontMetrics']);
        $domPdf->method('render')->willReturn(null);
        $domPdf->method('output')->willReturn(false);
        $domPdf->method('getFontMetrics')->willReturn($this->createStub(\Dompdf\FontMetrics::class));

        $template = $this->getMockForAbstractClass(
            AbstractTemplate::class,
            [Locale::NL_NL, $this->getMockForDataObject(), $this->getMockForTranslator(), $this->getResourcesPath()]
        );
        $template->method('getTemplate')->willReturn('foo-bar.php');

        $generator = new Generator($domPdf, $template);
        $generator->generate();
    }
}
