<?php

declare(strict_types=1);

namespace Dealerdirect\Agreements\Pdf\Templates;

use Dealerdirect\Agreements\Enums\Locale;
use Dealerdirect\Agreements\Utils\GeneralUtils;
use DealerDirect\Generic\Category\Site;

class CarSalesAgreement extends SalesAgreement
{
    public function getVehicleValues(): array
    {
        $data = [
            [
                'label' => $this->translate('labels.make_model'),
                'value' => mb_strtoupper("{$this->data->vehicle->brand} {$this->data->vehicle->model}"),
            ],
            [
                'label' => $this->translate('labels.vehicle_trim'),
                'value' => $this->data->vehicle->vehicleTrim,
            ],
            Locale::NL_NL === $this->data->locale ? [
                'label' => $this->translate('labels.license_plate'),
                'value' => $this->data->vehicle->licensePlate,
            ] : [
                'label' => $this->translate('labels.vin'),
                'value' => $this->data->vehicle->vin,
            ],
            [
                'label' => $this->translate('labels.old_new_model'),
                'value' => $this->translate("general.oldNewModel.{$this->data->vehicle->oldNewModel}"),
            ],
            [
                'label' => sprintf(
                    '%s / %s',
                    $this->translate('labels.construction_year'),
                    $this->translate('labels.first_admission')
                ),
                'value' => sprintf(
                    '%d / %s',
                    $this->data->vehicle->constructionYear,
                    $this->data->vehicle->firstAscriptionDate?->format('d-m-Y') ?? '-'
                ),
            ],
            [
                'label' => $this->translate('labels.odometer_reading'),
                'value' => $this->data->vehicle->getOdometerReading(),
            ],
            [
                'label' => $this->translate('labels.inspection_valid_until'),
                'value' => $this->data->vehicle->inspectionValidUntil?->format('d-m-Y') ?? '-',
            ],
            [
                'label' => "{$this->translate('labels.engine_capacity')} / {$this->translate('labels.engine_power')}",
                'value' => sprintf(
                    '%s cc / %s',
                    $this->data->vehicle->engineCapacity,
                    implode(', ', array_map(
                        fn ($enginePower) => sprintf('%d kW', $enginePower ?: 0),
                        $this->data->vehicle->enginePowers
                    )) ?: '0 kW'
                ),
            ],
            [
                'label' => $this->translate('labels.body_type'),
                'value' => $this->translate("vehicle.bodytype.{$this->data->vehicle->bodyType}"),
            ],
            [
                'label' => $this->translate('labels.fuel_type'),
                'value' => $this->translate("general.fuel.{$this->data->vehicle->fuel}"),
            ],
            [
                'label' => $this->translate('labels.transmission'),
                'value' => $this->translate("vehicle.transmission.{$this->data->vehicle->transmission}"),
            ],
            [
                'label' => "{$this->translate('labels.color')} / {$this->translate('labels.type')}",
                'value' => sprintf(
                    '%s / %s',
                    $this->translate("general.color.{$this->data->vehicle->colors[0]}"),
                    is_integer($this->data->vehicle->paintType)
                        ? $this->translate("general.paintworktype.{$this->data->vehicle->paintType}")
                        : '',
                ),
            ],
            [
                'label' => $this->translate('labels.upholstery'),
                'value' => empty($this->data->vehicle->upholstery)
                    ? $this->translate('vehicle.upholstery.0')
                    : $this->translate("vehicle.upholstery.{$this->data->vehicle->upholstery}"),
            ],
            [
                'label' => $this->translate('labels.margin_vat'),
                'value' => $this->translate("general.valueaddedtax.{$this->data->vehicle->marginVat}"),
            ],
        ];

        if (in_array($this->data->siteReference, [Site::REF_CAR_NL, Site::REF_CAR_MARKTPLAATS_NL])) {
            $data[] = [
                'label' => $this->translate('labels.registration_possible'),
                'value' => match ($this->data->vehicle->registrationPossible) {
                    1 => $this->translate('labels.yes'),
                    0 => $this->translate('labels.no'),
                    default => $this->translate('labels.unknown'),
                },
            ];
            $data[] = [
                'label' => $this->translate('labels.wok_notification'),
                'value' => match ($this->data->vehicle->wokNotification) {
                    1 => $this->translate('labels.yes'),
                    0 => $this->translate('labels.no'),
                    default => $this->translate('labels.unknown'),
                },
            ];
        }

        return $data;
    }

    public function getConsideration(): ?array
    {
        return null;
    }

    public function getMediation(): ?array
    {
        return null;
    }

    public function getConditions(): array
    {
        return [
            [
                'label' => $this->translate('labels.interior'),
                'value' => $this->translate("vehicle.condition.{$this->data->conditions->interior}"),
            ],
            [
                'label' => $this->translate('labels.exterior'),
                'value' => $this->translate("vehicle.condition.{$this->data->conditions->exterior}"),
            ],
            [
                'label' => $this->translate('labels.technical'),
                'value' => $this->translate("vehicle.condition.{$this->data->conditions->technical}"),
            ],
            [
                'label' => $this->translate('labels.tire_profile'),
                'value' => $this->translate("vehicle.condition.{$this->data->conditions->tireProfile}"),
            ],
            [
                'label' => $this->translate('labels.damage_free'),
                'value' => $this->translate("general.damageFree.{$this->data->conditions->damageFree}"),
            ],
            [
                'label' => $this->translate('labels.drivable'),
                'value' => $this->translate("general.driveable.{$this->data->conditions->drivable}"),
            ],
        ];
    }

    public function getAgreements(): array
    {
        return [
            'title' => $this->translate('labels.considering_that'),
            'parts' => [
                [
                    'title' => $this->translate('labels.ownership'),
                    'content' => $this->translate('sales_agreement.agreements.ownership', [
                        'brand' => $this->data->vehicle->brand,
                        'model' => $this->data->vehicle->model,
                    ]),
                ],
                [
                    'title' => $this->translate('labels.state'),
                    'content' => $this->translate('sales_agreement.agreements.status'),
                ],
                [
                    'title' => $this->translate('labels.vehicle'),
                    'content' => $this->translate('sales_agreement.agreements.description', [
                        'brand' => $this->data->vehicle->brand,
                        'model' => $this->data->vehicle->model,
                        'construction_year' => $this->data->vehicle->constructionYear,
                        'odometer_reading' => match ($value = $this->data->vehicle->getOdometerReading()) {
                            '-' => 0,
                            default => $value
                        },
                        'license_plate' => $this->data->vehicle->licensePlate,
                    ]),
                ],
                [
                    'title' => $this->translate('labels.payment'),
                    'content' => $this->translate('sales_agreement.agreements.payment', [
                        'sale_price' => number_format(
                            GeneralUtils::getConsumerPrice($this->data->salesPrice, $this->data->salesFee),
                            0,
                            ',',
                            '.'
                        ),
                    ]),
                ],
                [
                    'title' => $this->translate('labels.delivery'),
                    'content' => $this->data->pickUp
                        ? $this->translate('sales_agreement.agreements.delivery_pickup')
                        : $this->translate('sales_agreement.agreements.delivery_deliver'),
                ],
                [
                    'title' => $this->translate('labels.mediation'),
                    'content' => $this->translate('sales_agreement.agreements.mediation', [
                        'agent_name' => 0 !== $this->data->salesAgent->id ? $this->data->salesAgent->name : '',
                    ]),
                    //  nl2br($this->data->sales_agreement_mediation),
                ],
            ],
        ];
    }
}
