<?php

namespace Dealerdirect\Agreements\Agreements;

use Dealerdirect\Agreements\Interfaces\Model;
use Dealerdirect\Agreements\Storage;
use Dealerdirect\Agreements\Utils;
use DealerDirect\Generic\Category\EmailType;
use DealerDirect\Generic\Category\Locale;
use DealerDirect\Generic\Category\TranslationTypes;
use DealerDirect\Generic\Category\VehicleType;
use DealerDirect\Generic\Repository\CommonTranslator;
use DealerDirect\Generic\Repository\GenericProperties;
use DealerDirect\Generic\Repository\Translator;
use Dealerdirect\Sdk\Model\Email\Attachment;
use Dealerdirect\Sdk\Model\Email\Email;
use Dealerdirect\Sdk\Model\Email\Recipient;
use Dealerdirect\Sdk\Model\Email\SubstitutionKeys as Key;
use Dealerdirect\Sdk\Model\Email\Substitutions;
use mikehaertl\pdftk\Pdf;

final class SalesAgreement extends AbstractAgreement
{
    /**
     * @var string $saveFile
     */
    private $saveFile;
    /**
     * @var string $locale
     */
    private $locale;
    /**
     * @var string $vehicleType
     */
    private $vehicleType;

    /**
     * SalesAgreement constructor
     * Construct its parent and sets the data
     *
     * @param Model  $model
     * @param string $emailApiUrl
     * @param string $location
     */
    public function __construct(Model $model, string $emailApiUrl, string $location)
    {
        parent::__construct($model, $emailApiUrl);

        $this->location = $location;
        $this->data = $this->model->getSaleData();
        $this->locale = $this->model->getLocale();
        $this->vehicleType = $this->model->getVehicleType();
        $this->setDefaultData($model);
        $this->data['consumer_sale_price'] = $this->getConsumerPrice();
        $this->saveFile = sprintf(
            'verkoopovereenkomsten/%s.pdf',
            sha1(
                sprintf(
                    '%s%s%s%s',
                    $this->data['company_id'],
                    $this->data['sales_date'],
                    $this->model->getDeliverId(),
                    $this->data['make']
                )
            )
        );
    }

    /**
     * @return string
     */
    private function getConsumerPrice(): string
    {
        $saleAmountConsumer = round($this->data['sales_price'] - $this->data['sales_fee']);

        return number_format($saleAmountConsumer, 0, ",", ".");
    }

    /**
     * @return string
     */
    private function getPdfName(): string
    {
        $name = 'Verkoopovereenkomst_%d.pdf';
        if ($this->locale === Locale::DE_DE) {
            $name = 'Verkaufbestaetigung_%d.pdf';
        }

        return sprintf($name, $this->data['lot_id']);
    }

    /**
     * Creates the SalesAgreement
     *
     * @return bool
     */
    public function create(): bool
    {
        // If agreement already exists then don't generate a new one.
        if (Storage::has($this->location, $this->saveFile)) {
            return true;
        }
        $requiredFileName = sprintf(
            '%s-%s',
            $this->vehicleType,
            strtolower($this->locale)
        );

        $vehicleInfo = $this->getTranslations();
        $lang = include dirname(__DIR__) . "/resources/lang/{$requiredFileName}.php";
        $pdfFile = dirname(__DIR__) . "/resources/files/sales-agreements/{$requiredFileName}.pdf";

        $pdf = new Pdf($pdfFile);
        $pdfCreated = $pdf->fillForm($lang)
            ->needAppearances()
            ->flatten()
            ->allow('Printing CopyContents ScreenReaders')
            ->setPassword(base64_encode(random_bytes(64))) // Set owner password
            ->passwordEncryption(128) // Set password encryption strength
            ->execute();

        try {
            if ($pdfCreated !== true) {
                throw new \Exception("Could not open file. Message = '{$pdf->getError()}'");
            }
            $stream = fopen($pdf->getTmpFile(), 'r+');
            $storeResult = Storage::putStream($this->location, $this->saveFile, $stream);
        } finally {
            if (isset($stream) && is_resource($stream)) {
                fclose($stream);
            }
        }

        return $storeResult ?? false;
    }

    /**
     * Sends the SalesAgreement to email-service
     *
     * @return boolean
     */
    public function send(): bool
    {
        $email = new Email(
            $this->locale,
            $this->vehicleType,
            EmailType::SALE_AGREEMENT,
            $this->getSubstitutions(),
            $this->getAttachments()
        );
        $bcc = [];
        if (Utils::isMotor($this->vehicleType)) {
            $bcc[] = new Recipient('kerim@dealerdirect.nl', 'Kerim Gilbers');
        }

        return $this->sendEmail(
            $email,
            [new Recipient($this->data['consumer_email'], $this->data['consumer_name'])],
            [],
            $bcc
        );
    }

    private function getAttachments(): array
    {
        $attachments = [
            new Attachment(
                Storage::read($this->location, $this->saveFile),
                'application/pdf',
                $this->getPdfName()
            )
        ];

        $vehicleType = $this->vehicleType;
        if ($this->locale === Locale::NL_NL && $vehicleType !== VehicleType::BICYCLE) {
            $filesPath = dirname(__DIR__) . '/resources/files';

            $file = "{$filesPath}/Controlelijst_IWVMAA.pdf";
            if ($vehicleType === VehicleType::MOTOR) {
                $file = "{$filesPath}/SD-00414_Kentekendocumenten_Overdracht.pdf";
            } elseif ($vehicleType === VehicleType::SCOOTER) {
                $file = "{$filesPath}/SD-00414_Kentekendocumenten_Overdracht_ScooterNL.pdf";
            }

            $attachments[] = new Attachment(
                file_get_contents($file),
                'application/pdf',
                'Documenten_Overdracht.pdf'
            );
        }

        return $attachments;
    }

    /**
     * Gets the substitutions for the email to be send.
     *
     * @return Substitutions
     */
    private function getSubstitutions(): Substitutions
    {
        $data = [
            Key::CONSUMER_ADDRESS => $this->getConsumerAddress(),
            Key::CONSUMER_EMAIL => $this->data['consumer_email'],
            Key::CONSUMER_PHONE => Utils::combinePhoneNumbers(
                $this->data['consumer_phone'],
                $this->data['consumer_mobile']
            ),

            Key::DEALER_ADDRESS => $this->getCompanyAddress(),
            Key::DEALER_EMAIL => $this->data['company_agree_email'],
            Key::DEALER_NAME => $this->data['company_name'],
            Key::DEALER_PHONE => $this->data['company_phone'],

            Key::FIRST_NAME => $this->data['consumer_first_letters'],
            Key::LAST_NAME => $this->data['consumer_name'],
            Key::LOT_NUMBER => $this->data['lot_id'],
            Key::VEHICLE_MAKE => $this->data['make'],
            Key::VEHICLE_MODEL => $this->data['model'],
        ];
        if (Utils::isMotor($this->vehicleType)) {
            $data[Key::AGENT_PHONE] = Utils::getAgentPhoneByReference($this->data['site_reference']);
            if ($this->locale === Locale::DE_DE) {
                $data[Key::VEHICLE_CONSUMER_PRICE] = $this->data['consumer_sale_price'];
                $data[Key::SALUTATION] = Utils::getSalutationByGender(
                    $this->locale,
                    (int) $this->data['consumer_salutation']
                );
            }
        }
        return new Substitutions($data);
    }

    /**
     * @return array
     */
    private function translateDefaults(): array
    {
        $vehicleInfo = $this->data;
        $vehicleTranslator = $this->translator;
        $vehicleTranslator->getAll(TranslationTypes::VEHICLE_VALUES);
        $generalTranslator = new CommonTranslator($this->genericProperties->getLocaleInstance());
        $generalTranslator->getAll(TranslationTypes::GENERAL_VALUES);
        $mappings = [
            TranslationTypes::VEHICLE_VALUES => [
                'upholstery' => 'upholstery',
                'transmission' => 'transmission',
                'bodywork' => 'bodytype',
                'condition_bodywork' => 'condition',
                'condition_interior' => 'condition',
                'condition_technical' => 'condition',
                'condition_tyre_profile' => 'condition',
                'condition_exterior' => 'condition',
                'condition_front_tyre' => 'condition',
                'condition_propulsion' => 'condition',
                'condition' => 'condition',
                'condition_battery' => 'condition',

            ],
            TranslationTypes::GENERAL_VALUES => [
                'fuel' => 'fuel',
                'color' => 'color',
                'color_2' => 'color',
                'color_3' => 'color',
                'paint_type' => 'paintworktype',
                'marge' => 'valueaddedtax',
                'power_2' => 'enginepower',
                'mileage_type' => 'distanceunit',
            ],
        ];

        foreach ($mappings as $translationType => $translatable) {
            $translator = $translationType === TranslationTypes::VEHICLE_VALUES
                ? $vehicleTranslator
                : $generalTranslator;

            foreach ($translatable as $key => $valKey) {
                if (
                    array_key_exists($key, $vehicleInfo)
                    && $translator->isTranslated($valKey, $vehicleInfo[$key])
                ) {
                    $vehicleInfo[$key] = $translator->translate($valKey, $vehicleInfo[$key]);
                }
            }
        }

        return $vehicleInfo;
    }

    /**
     * Translates all the values from $this->data
     *
     * @return array
     */
    private function getTranslations(): array
    {
        $vehicleInfo = $this->translateDefaults();
        $vehicleInfo['consumer_country_name'] = $this->countryIdToTranslated((int) $vehicleInfo['consumer_country']);
        $vehicleInfo['company_country_name'] = $this->countryIdToTranslated((int) $vehicleInfo['company_country']);
        $vehicleInfo['options'] = $this->splitOptions(
            $this->translateOptions((string) $vehicleInfo['vehicle_options'])
        );

        if (Utils::isMotor($this->vehicleType)) {
            foreach (['color_2', 'color_3'] as $key) {
                $vehicleInfo[$key] = $vehicleInfo[$key] === '0' ? '-' : $vehicleInfo[$key];
            }
        }
        foreach (['condition_damage_free', 'condition_drivable'] as $key) {
            if (array_key_exists($key, $vehicleInfo)) {
                $vehicleInfo[$key] = Utils::translateUnknownYesNo($vehicleInfo[$key], $this->locale);
            }
        }
        foreach (['consumer', 'company'] as $type) {
            $vehicleInfo["{$type}_phone"] = Utils::sanitizePhoneNumber(
                (int) $vehicleInfo["{$type}_country"],
                $vehicleInfo["{$type}_phone"]
            );
            if (array_key_exists("{$type}_mobile", $vehicleInfo)) {
                $vehicleInfo["{$type}_mobile"] = Utils::sanitizePhoneNumber(
                    (int) $vehicleInfo["{$type}_country"],
                    $vehicleInfo["{$type}_mobile"]
                );
            }
        }

        return $vehicleInfo;
    }

    /**
     * Splits the vehicle options in two rows.
     *
     * @param  array $options
     * @return array
     */
    private function splitOptions(array $options): array
    {
        $optionList[0] = false;
        $optionList[1] = false;
        if (!empty($options)) {
            $splitOptions = array_chunk($options, ceil(count($options) / 2));
            $optionList[0] = implode(PHP_EOL, $splitOptions[0]);
            if (!empty($splitOptions[1])) {
                $optionList[1] = implode(PHP_EOL, $splitOptions[1]);
            }
        }

        return $optionList;
    }

    /**
     * Translates the options, and includes recreational options if the vehicleType is CAR.
     *
     * @param  string $options
     * @return array
     */
    private function translateOptions(string $options): array
    {
        $allOptions = $this->translator->getAll(TranslationTypes::OPTIONS);
        if ($this->vehicleType === VehicleType::CAR) {
            $locale = $this->genericProperties->getLocaleInstance();
            $recreationGenericProperties = new GenericProperties($locale, VehicleType::RECREATION);
            $recreationTranslator = new Translator($recreationGenericProperties);
            $allOptions = $allOptions + $recreationTranslator->getAll(TranslationTypes::OPTIONS);
        }

        if (!is_array($options)) {
            $options = explode(';', trim($options, ';'));
        }

        return array_intersect_key($allOptions, array_flip($options));
    }
}
