<?php

declare(strict_types=1);

namespace Dealerdirect\Agreements\Pdf;

use Dealerdirect\Agreements\Exceptions\PdfGenerationException;
use Dealerdirect\Agreements\Pdf\Templates\AbstractTemplate;
use Dompdf\Dompdf;

final class Generator
{
    public function __construct(
        private Dompdf $domPdf,
        private AbstractTemplate $template
    ) {
        $this->setFonts();
    }

    public function getTemplate(): AbstractTemplate
    {
        return $this->template;
    }

    public function generate(): string
    {
        $this->domPdf->loadHtml($this->template->render());
        $this->domPdf->setPaper(
            $this->template->getPaperSize()->value,
            $this->template->getOrientation()->value
        );

        $this->domPdf->render();
        if (empty($pdf = $this->domPdf->output())) {
            throw new PdfGenerationException($this->template);
        }

        return $pdf;
    }

    public static function fromTemplate(AbstractTemplate $template): self
    {
        $domPdf = new Dompdf();

        return new self($domPdf, $template);
    }

    private function setFonts(): void
    {
        $resourcesFolder = $this->template->getResourcesFolder();

        $options = $this->domPdf->getOptions();
        $options->setChroot([$options->getRootDir(), $resourcesFolder]);
        $options->setFontDir("$resourcesFolder/fonts/cache");
        $options->setFontCache("$resourcesFolder/fonts/cache");

        $this->domPdf->setOptions($options);

        $fontMetrics = $this->domPdf->getFontMetrics();

        foreach ($this->template->getFonts() as $font) {
            $font->register($fontMetrics);
        }

        $this->domPdf->setFontMetrics($fontMetrics);
    }
}
