<?php

declare(strict_types=1);

namespace Dealerdirect\Agreements\Tests\Pdf\Templates;

use Dealerdirect\Agreements\Enums\Orientation;
use Dealerdirect\Agreements\Enums\PaperSize;
use Dealerdirect\Agreements\Exceptions\TemplateFileNotFoundException;
use Dealerdirect\Agreements\Exceptions\TemplateRenderingException;
use Dealerdirect\Agreements\Pdf\Templates\AbstractTemplate;
use Dealerdirect\Agreements\Tests\TestCase;

/**
 * @covers \Dealerdirect\Agreements\Pdf\Templates\AbstractTemplate
 * @covers \Dealerdirect\Agreements\Enums\Locale
 *
 * @uses \Dealerdirect\Agreements\Utils\Translator
 */
final class AbstractTemplateTest extends TestCase
{
    private AbstractTemplate $template;

    public function setUp(): void
    {
        parent::setUp();

        $this->template = $this->getTemplateMock(
            $this->getMockForDataObject(['foo' => 'bar']),
            $this->getTranslatorInstance(),
        );
    }

    public function test_get_paper_size_returns_default_value(): void
    {
        $this->assertSame(PaperSize::A4, $this->template->getPaperSize());
        $this->assertSame('A4', $this->template->getPaperSize()->value);
    }

    public function test_get_orientation_returns_default_value(): void
    {
        $this->assertSame(Orientation::PORTRAIT, $this->template->getOrientation());
        $this->assertSame('portrait', $this->template->getOrientation()->value);
    }

    /**
     * @covers \Dealerdirect\Agreements\Exceptions\TemplateFileNotFoundException
     */
    public function test_render_should_throw_exception_when_template_file_not_found(): void
    {
        $this->expectException(TemplateFileNotFoundException::class);

        call_user_func([$this->template, 'setTemplate'], 'a-non-existing-template-file');

        $this->template->render();
    }

    public function test_render_should_return_rendered_template(): void
    {
        $this->assertSame(
            "<html><body>Hello, bar!</body></html>\n",
            $this->template->render()
        );
    }

    /**
     * @covers \Dealerdirect\Agreements\Exceptions\TemplateRenderingException
     */
    public function test_render_should_throw_exception_when_template_has_errors(): void
    {
        $this->expectException(TemplateRenderingException::class);

        call_user_func([$this->template, 'setTemplate'], 'invalid.php');

        $this->template->render();
    }

    public function test_get_resources_folder(): void
    {
        $this->assertSame(
            $this->getResourcesPath(),
            $this->template->getResourcesFolder()
        );
    }
}
