<?php

declare(strict_types=1);

namespace Dealerdirect\Agreements\Tests\Utils;

use Dealerdirect\Agreements\Tests\TestCase;
use Dealerdirect\Agreements\Utils\CommentsFormatter;
use Dealerdirect\Generic\Category\Driveable;

/**
 * @covers \Dealerdirect\Agreements\Utils\CommentsFormatter
 */
final class CommentsFormatterTest extends TestCase
{
    /**
     * @dataProvider commentsProvider
     * @param array<string,string> $expectedResult
     */
    public function test_format_returns_expected_result(
        string $internalComment,
        string $consumerComment,
        array $expectedResult
    ): void {
        $formatter = new CommentsFormatter($internalComment, $consumerComment);
        $actual = $formatter->format();

        foreach ($expectedResult as $key => $value) {
            $this->assertSame($value, $actual[$key], sprintf(
                'Expected %s to be %s, got %s',
                $key,
                $this->displayValue($value),
                $this->displayValue($actual[$key]),
            ));
        }
    }

    /**
     * @return array<int,array<int,string|array<string,string>>>
     */
    public static function commentsProvider(): array
    {
        return [
            0 => [
                <<<TXT
                    VIN: 1234567890

                    Carrosserie

                    Zijn er gebruikerssporen aan de carrosserie? Ja
                    TXT,
                '',
                [
                    'vin' => '1234567890',
                    'drivable' => Driveable::UNKNOWN,
                    'consumer_comments' => '',
                    'internal_comments' => <<<TXT
                        Carrosserie

                        Zijn er gebruikerssporen aan de carrosserie? Ja
                        TXT,
                ],
            ],
            1 => [
                <<<TXT
                    Start de motor?: Ja
                    VIN: VG5DM01B000000577
                    TXT,
                'er is een klein stukje van het windscherm af en er zijn lichte krasjes aan de buitenkant van de'
                    . ' spiegels',
                [
                    'drivable' => Driveable::YES,
                    'vin' => 'VG5DM01B000000577',
                    'consumer_comments' => 'er is een klein stukje van het windscherm af en er zijn lichte krasjes aan'
                        . ' de buitenkant van de spiegels',
                    'internal_comments' => '',
                ],
            ],
            2 => [
                '',
                '',
                [
                    'drivable' => Driveable::UNKNOWN,
                    'vin' => null,
                    'consumer_comments' => '',
                    'internal_comments' => '',
                ],
            ],
            3 => [
                'Ist das Motorrad fahrtüchtig?: Nein',
                "\n\n Some text will strip the line breaks and whitespace at the beginning and end \n\n",
                [
                    'drivable' => Driveable::NO,
                    'vin' => null,
                    'consumer_comments' => 'Some text will strip the line breaks and whitespace at the beginning and'
                        . ' end',
                    'internal_comments' => '',
                ],
            ],
        ];
    }
}
