<?php

declare(strict_types=1);

namespace Dealerdirect\Agreements\DataObjects;

use Dealerdirect\Agreements\Enums\Locale;
use Dealerdirect\Agreements\Utils\GeneralUtils;
use DealerDirect\Generic\Category\DistanceUnit;

final class Vehicle implements DataObject
{
    /**
     * @param int[] $enginePowers
     * @param int[] $colors
     * @param int[] $options
     * @param null|array{questions:array,answers:array} $checklist
     */
    public function __construct(
        public readonly ?string $brand,
        public readonly ?string $model,
        public readonly ?int $bodyType,
        public readonly ?string $vehicleTrim,
        public readonly ?int $engineCapacity,
        public readonly array $enginePowers,
        public readonly ?int $cylinderAmount,
        public readonly ?int $fuel,
        public readonly ?int $transmission,
        public readonly ?int $odometerReading,
        public readonly ?int $odometerReadingUnit,
        public readonly int $constructionYear,
        public readonly array $colors,
        public readonly ?\DateTime $inspectionValidUntil,
        public readonly ?\DateTime $firstAscriptionDate,
        public readonly ?int $paintType,
        public readonly ?int $upholstery,
        public readonly ?int $marginVat,
        public readonly array $options,
        public readonly ?string $licensePlate,
        public readonly ?string $vin,
        public readonly ?int $registrationPossible,
        public readonly ?int $wokNotification,
        public readonly ?int $maintenanceHistory,
        public readonly ?int $bicycleType,
        public readonly ?int $enginePosition,
        public readonly ?int $batteryWattage,
        public readonly ?int $amountOfGears,
        public readonly ?int $oldNewModel,
        public readonly ?string $image,
        public readonly Conditions $conditions,
        private readonly ?array $checklist,
    ) {
    }

    public function getOdometerReading(bool $withUnit = true): string
    {
        if (null === $this->odometerReading) {
            return '-';
        }

        $result = GeneralUtils::formatNumber($this->odometerReading);
        if ($withUnit) {
            $result .= ' ' . (DistanceUnit::MILES === $this->odometerReadingUnit ? 'mi' : 'km');
        }

        return $result;
    }

    public function getChecklist(Locale $locale): ?array
    {
        if (empty($this->checklist)) {
            return null;
        }

        $result = [];
        $questions = $this->checklist['questions'];
        $answers = $this->checklist['answers'];

        foreach ($questions as $question) {
            $localizedQuestions = json_decode($question['question'], true, flags: JSON_THROW_ON_ERROR);
            $options = json_decode($question['options'], true, flags: JSON_THROW_ON_ERROR);
            $localizedAnswers = $options['answers'] ?? [];

            $answer = $answers[$question['id']] ?? null;
            if ('date' === $options['type']) {
                $answer = GeneralUtils::formatDateOrDefault($answer);
            } elseif (isset($answer['answer'])) {
                $answer['answer'] = $localizedAnswers[$answer['answer']][$locale->value] ?? $answer['answer'];
            }

            $result[] = [
                'question' => $localizedQuestions[$locale->value],
                'answer' => GeneralUtils::formatSurveyAnswer($answer),
            ];
        }

        return $result;
    }
}
