<?php

declare(strict_types=1);

namespace Dealerdirect\Agreements\Pdf\Templates;

use Dealerdirect\Agreements\Enums\Locale;
use Dealerdirect\Agreements\Utils\GeneralUtils;
use DealerDirect\Generic\Category\Site;

class MotorbikeSalesAgreement extends SalesAgreement
{
    public function getConsideration(): ?array
    {
        return [
            'title' => $this->translate('labels.considering_that'),
            'content' => $this->translate('sales_agreement.ownership', [
                'brand' => $this->data->vehicle->brand,
                'model' => $this->data->vehicle->model,
            ]),
        ];
    }

    public function getMediation(): ?array
    {
        return [
            'title' => $this->translate('labels.mediation'),
            'content' => $this->translate('sales_agreement.mediation', [
                'agent_name' => !empty($this->data->salesAgent->id) ? $this->data->salesAgent->name : '',
            ]),
        ];
    }

    public function getVehicleValues(): array
    {
        $data = array_filter([
            [
                'label' => $this->translate('labels.make_model'),
                'value' => "{$this->data->vehicle->brand} {$this->data->vehicle->model}",
            ],
            [
                'label' => $this->translate('labels.vehicle_trim'),
                'value' => $this->data->vehicle->vehicleTrim,
            ],
            Locale::DE_DE === $this->data->locale ? [
                'label' => $this->translate('labels.construction_year'),
                'value' => $this->data->vehicle->constructionYear,
            ] : [
                'label' => sprintf(
                    '%s / %s',
                    $this->translate('labels.construction_year'),
                    $this->translate('labels.first_admission')
                ),
                'value' => sprintf(
                    '%d / %s',
                    $this->data->vehicle->constructionYear,
                    $this->data->vehicle->firstAscriptionDate?->format('d-m-Y') ?? '-',
                ),
            ],
            [
                'label' => $this->translate('labels.odometer_reading'),
                'value' => $this->data->vehicle->getOdometerReading(),
            ],
            Locale::NL_NL === $this->data->locale ? [
                'label' => $this->translate('labels.license_plate'),
                'value' => $this->data->vehicle->licensePlate,
            ] : null,
            [
                'label' => $this->translate('labels.vin'),
                'value' => $this->data->vehicle->vin,
            ],
            [
                'label' => $this->translate('labels.engine_power'),
                'value' => implode(', ', array_map(
                    fn ($enginePower) => "$enginePower kW",
                    $this->data->vehicle->enginePowers
                )) ?: '0 kW',
            ],
            [
                'label' => $this->translate('labels.body_type'),
                'value' => $this->translate("vehicle.bodytype.{$this->data->vehicle->bodyType}"),
            ],
            [
                'label' => $this->translate('labels.fuel_type'),
                'value' => $this->translate("general.fuel.{$this->data->vehicle->fuel}"),
            ],
            [
                'label' => $this->translate('labels.transmission'),
                'value' => $this->translate("vehicle.transmission.{$this->data->vehicle->transmission}"),
            ],
            [
                'label' => $this->translate('labels.colors'),
                'value' => implode(' / ', array_map(
                    fn ($color) => $this->translate("general.color.{$color}"),
                    $this->data->vehicle->colors
                )),
            ],
            Locale::DE_DE !== $this->data->locale ? [
                'label' => $this->translate('labels.margin_vat'),
                'value' => $this->translate("general.valueaddedtax.{$this->data->vehicle->marginVat}"),
            ] : null,
        ]);

        if (in_array($this->data->siteReference, [Site::REF_MOTO_NL, Site::REF_SCOOTER_NL])) {
            $data[] = [
                'label' => $this->translate('labels.registration_possible'),
                'value' => match ($this->data->vehicle->registrationPossible) {
                    1 => $this->translate('labels.yes'),
                    0 => $this->translate('labels.no'),
                    default => $this->translate('labels.unknown'),
                },
            ];
            $data[] = [
                'label' => $this->translate('labels.wok_notification'),
                'value' => match ($this->data->vehicle->wokNotification) {
                    1 => $this->translate('labels.yes'),
                    0 => $this->translate('labels.no'),
                    default => $this->translate('labels.unknown'),
                },
            ];
        }

        return $data;
    }

    public function getAgreements(): array
    {
        return [
            'title' => $this->translate('labels.buyer_seller_agreement'),
            'parts' => [
                [
                    'title' => $this->translate('labels.vehicle'),
                    'content' => $this->translate('sales_agreement.agreements.vehicle', [
                        'brand' => $this->data->vehicle->brand,
                        'model' => $this->data->vehicle->model,
                        'construction_year' => $this->data->vehicle->constructionYear,
                        'odometer_reading' => match ($value = $this->data->vehicle->getOdometerReading()) {
                            '-' => 0,
                            default => $value,
                        },
                        'license_plate' => $this->data->vehicle->licensePlate,
                    ]),
                ],
                [
                    'title' => $this->translate('labels.payment'),
                    'content' => $this->translate('sales_agreement.agreements.payment', [
                        'sale_price' => number_format(GeneralUtils::getConsumerPrice(
                            $this->data->salesPrice,
                            $this->data->salesFee
                        ), 0, ',', '.'),
                    ]),
                ],
                [
                    'title' => $this->translate('labels.delivery'),
                    'content' => $this->data->pickUp
                        ? $this->translate('sales_agreement.agreements.delivery_pickup')
                        : $this->translate('sales_agreement.agreements.delivery_deliver'),
                ],
            ],
        ];
    }
}
