<?php

namespace Dealerdirect\Agreements\Agreements;

use Dealerdirect\Agreements\Agreements\AbstractAgreement;
use Dealerdirect\Agreements\Interfaces\Model;
use Dealerdirect\Agreements\Utils;
use DealerDirect\Generic\Category\CallGroupsMoto;
use DealerDirect\Generic\Category\EmailType;
use DealerDirect\Generic\Category\Locale;
use DealerDirect\Generic\Category\TranslationTypes;
use DealerDirect\Generic\Category\VehicleType;
use DealerDirect\Generic\Repository\CallGroupInformation;
use Dealerdirect\Sdk\Model\Email\Attachment;
use Dealerdirect\Sdk\Model\Email\Email;
use Dealerdirect\Sdk\Model\Email\Recipient;
use Dealerdirect\Sdk\Model\Email\SubstitutionKeys as Key;
use Dealerdirect\Sdk\Model\Email\Substitutions;
use Exception;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\BadResponseException;
use GuzzleHttp\Psr7\Request;

final class PurchaseAgreement extends AbstractAgreement
{
    /**
     * @var string $impersonateKey
     */
    private $impersonateKey;
    /**
     * @var string $pdfName
     */
    protected $pdfName = 'Aankoopspecificatie - kavel %d.pdf';
    /**
     * @var string $dealerdirectUrl
     */
    private $dealerdirectUrl;
    /**
     * @var string $locale
     */
    private $locale;
    /**
     * @var string $vehicleType
     */
    private $vehicleType;
    /**
     * @var string $pdfContent
     */
    private $pdfContent;

    /**
     * PurchaseAgreement constructor
     * {@inheritDoc}
     *
     * @param string $impersonateKey
     * @param string $dealerdirectUrl
     */
    public function __construct(Model $model, string $emailApiUrl, string $impersonateKey, string $dealerdirectUrl)
    {
        parent::__construct($model, $emailApiUrl);

        $this->data = $this->model->getPurchaseData();
        $this->locale = $this->model->getLocale();
        $this->vehicleType = $this->model->getVehicleType();
        $this->impersonateKey = $impersonateKey;
        $this->dealerdirectUrl = $dealerdirectUrl;
        $this->setDefaultData($model);
    }

    private function getPdfUrl(string $language)
    {
        return sprintf(
            '%s/%s/purchase/%s/%d/pdf',
            $this->dealerdirectUrl,
            $language,
            $this->vehicleType,
            $this->data['lot_id']
        );
    }

    public function getPdfContent()
    {
        return $this->pdfContent;
    }

    /**
     * @return void
     */
    public function preview()
    {
        if ($this->create()) {
            header("Content-type:application/pdf");
            echo $this->pdfContent;
            exit;
        }

        echo 'No data!';
    }

    /**
     * Creates/fetches the pdf from dealerdirect.eu
     *
     * @return bool
     */
    public function create(): bool
    {
        $vehicleType = $this->vehicleType;
        $language = $this->getLanguageByLocale();
        $client = new Client(['allow_redirects' => true]);

        $request = new Request(
            'GET',
            $this->getPdfUrl($language),
            [
                'Accept-Language' => $language,
                'Authorization' => "Bearer {$this->impersonateKey}",
                'BedrijvenID' => (int) $this->data['company_id'],
                'Vehicle-Type' => Utils::isMotor($vehicleType) ? 'motobike' : $vehicleType,
            ]
        );

        return $this->sendRequest($client, $request);
    }

    /**
     * Sends the request
     *
     * @param  Client  $client
     * @param  Request $request
     * @return boolean
     * @throws BadResponseException
     */
    private function sendRequest(Client $client, Request $request): bool
    {
        try {
            $response = $client->send($request);
            if ($response->getStatusCode() === 200) {
                $this->pdfContent = (string) $response->getBody();
                $result = true;
            } else {
                throw new Exception(sprintf(
                    'Failed to retrieve pdf (lotId: %d, dealerId %d, Locale: %s [%s])',
                    $this->data['lot_id'],
                    $this->data['company_id'],
                    $this->locale,
                    strtoupper($this->vehicleType)
                ));
            }
        } catch (Exception $exception) {
            if (!extension_loaded('newrelic')) {
                throw $exception;
            }

            newrelic_notice_error(null, $exception);
        }

        return $result ?? false;
    }

    /**
     * Sends the agreement to email-service
     *
     * @return bool
     */
    public function send(): bool
    {
        $email = new Email(
            $this->locale,
            $this->vehicleType,
            EmailType::PURCHASE_AGREEMENT,
            $this->getSubstitutions(),
            $this->createAttachments()
        );

        $cc = [];
        $bcc = [];
        if (in_array($this->vehicleType, [VehicleType::SCOOTER, VehicleType::MOTOR], true)) {
            $bcc[] = new Recipient('kerim@dealerdirect.nl', 'Kerim Gilbers');
        } elseif ($this->locale === Locale::NL_NL && $this->data['sales_fee'] >= 500) {
            $bcc[] = new Recipient('feedback@ikwilvanmijnautoaf.nl', $this->data['company_name']);
        } elseif (
            $this->vehicleType === VehicleType::BICYCLE
            && $this->getCallGroupId() !== CallGroupsMoto::NL_BICYCLE_B2B
        ) {
            $cc[] = new Recipient('info@bhtl.nl', 'BHTL');
        }

        return $this->sendEmail($email, $this->getToRecipients(), $cc, $bcc);
    }

    private function getCallGroupId(): int
    {
        return (new CallGroupInformation($this->model->dbal, $this->vehicleType))
            ->getCallGroupId($this->data['lot_id']);
    }

    /**
     * Validates and gets the TO recipients
     *
     * @return array
     */
    private function getToRecipients(): array
    {
        $to = [];
        if (filter_var($this->data['company_agree_email'], FILTER_VALIDATE_EMAIL)) {
            $to[] = new Recipient($this->data['company_agree_email'], $this->data['company_name']);
        }
        if (
            $this->data['company_agree_email_2'] !== $this->data['company_agree_email']
            && filter_var($this->data['company_agree_email_2'], FILTER_VALIDATE_EMAIL)
        ) {
            $to[] = new Recipient($this->data['company_agree_email_2'], $this->data['company_name']);
        }

        return $to;
    }

    /**
     * Gets the pdf name created.
     * protected $pdfName must be set in extended class.
     *
     * @return string
     */
    private function getPdfName(): string
    {
        $names = [
            Locale::NL_NL => 'Aankoopspecificatie - kavel %d.pdf',
            Locale::NL_BE => 'Aankoopspecificatie - kavel %d.pdf',
            Locale::FR_BE => 'Spécification d\'achat - Lot %d.pdf',
            Locale::DE_DE => 'Kennziffer %d - Alle Daten im Überblick.pdf',
        ];
        $name = $names[$this->locale] ?? $names[Locale::NL_NL];

        return sprintf($name, $this->data['lot_id']);
    }

    /**
     * Gets all substitutions for email to send
     *
     * @return Substitutions
     */
    private function getSubstitutions(): Substitutions
    {
        $vehicleValues = $this->translator->getAll(TranslationTypes::VEHICLE_VALUES);
        $bodyworks = $vehicleValues['bodytype'];

        return new Substitutions(
            [
                Key::AGENT_NAME => $this->data['sales_agent'],
                Key::CONSUMER_ADDRESS => $this->getConsumerAddress(),
                Key::CONSUMER_NAME => sprintf(
                    '%s %s',
                    $this->data['consumer_first_letters'],
                    $this->data['consumer_name']
                ),
                Key::CONSUMER_EMAIL => $this->data['consumer_email'],
                Key::CONSUMER_PHONE => Utils::combinePhoneNumbers(
                    $this->data['consumer_phone'],
                    $this->data['consumer_mobile']
                ),
                Key::FULL_NAME => $this->data['company_name'],
                Key::LOT_NUMBER => $this->data['lot_id'],

                Key::VEHICLE_BODYWORK => $bodyworks[$this->data['bodywork']] ?? '-',
                Key::VEHICLE_CONSTRUCTION_YEAR => $this->data['construction_year'],
                Key::VEHICLE_CONSUMER_PRICE => $this->data['sales_price'] - $this->data['sales_fee'],
                Key::VEHICLE_DEALER_FEE => $this->data['sales_fee'],
                Key::VEHICLE_DEALER_PICK_UP => $this->data['pick_up'] ? 'ja' : 'nee',
                Key::VEHICLE_GROSS_PRICE => $this->data['sales_price'],
                Key::VEHICLE_MAKE => $this->data['make'],
                Key::VEHICLE_MODEL => $this->data['model'],
                Key::VEHICLE_MILEAGE => $this->data['mileage'],
                Key::VEHICLE_REGISTRATION_PLATE => $this->data['license_plate'],
                Key::VEHICLE_TRIM_LEVEL => $this->data['execution'],
                Key::VEHICLE_DELIVERY_DATE => (new \DateTime("{$this->data['sales_date']} + 14 days"))->format('d-m-Y'),

                Key::LEAD_CONSUMER_LABEL_NAME => $this->data['label_name'],
            ]
        );
    }

    /**
     * Checks and returns an array with attachments or an empty array
     *
     * @return array
     */
    private function createAttachments()
    {
        $attachments = [];
        if ($this->pdfContent) {
            $attachments[] = new Attachment(
                $this->pdfContent,
                'application/pdf',
                $this->getPdfName()
            );
        }

        return $attachments;
    }

    /**
     * Gets the language needed by dealerdirect.eu
     *
     * @return string
     */
    private function getLanguageByLocale()
    {
        $languages = [
            Locale::NL_NL => 'nl',
            Locale::NL_BE => 'nl-be',
            Locale::FR_BE => 'fr',
            Locale::DE_DE => 'de',
        ];

        return $languages[$this->locale] ?? 'en';
    }
}
