<?php

declare(strict_types=1);

namespace Dealerdirect\Agreements\Tests\Factories;

use Dealerdirect\Agreements\Enums\DealerdirectLanguage;
use Dealerdirect\Agreements\Exceptions\AllowRedirectsMustBeEnabledException;
use Dealerdirect\Agreements\Exceptions\FetchPurchaseAgreementPdfException;
use Dealerdirect\Agreements\Exceptions\PurchaseAgreementNotStoredException;
use Dealerdirect\Agreements\Factories\PurchaseAgreementFactory;
use Dealerdirect\Agreements\Tests\TestCase;
use Dealerdirect\Agreements\Tests\Traits\DataObjectFixtures;
use DealerDirect\Generic\Category\VehicleType;
use DealerDirect\Sdk\Facade\Email;
use GuzzleHttp\Client;
use GuzzleHttp\Psr7\Request;
use GuzzleHttp\Psr7\Response;
use Illuminate\Filesystem\FilesystemAdapter;
use League\Flysystem\Filesystem;
use League\Flysystem\Local\LocalFilesystemAdapter;

/**
 * @covers \Dealerdirect\Agreements\Factories\PurchaseAgreementFactory
 *
 * @uses \Dealerdirect\Agreements\DataObjects\Contact
 * @uses \Dealerdirect\Agreements\DataObjects\PurchaseAgreement
 * @uses \Dealerdirect\Agreements\Enums\Locale
 * @uses \Dealerdirect\Agreements\Utils\GeneralUtils
 */
final class PurchaseAgreementFactoryTest extends TestCase
{
    use DataObjectFixtures;

    private FilesystemAdapter $storage;

    public function setUp(): void
    {
        parent::setUp();

        $localFileSystem = new LocalFilesystemAdapter('/tmp/testing');
        $fileSystem = new Filesystem($localFileSystem);

        $this->storage = new FilesystemAdapter($fileSystem, $localFileSystem);
    }

    public function tearDown(): void
    {
        $this->storage->deleteDirectory('purchase-agreements');
        $this->storage->createDirectory('purchase-agreements');

        parent::tearDown();
    }

    /**
     * @covers \Dealerdirect\Agreements\Exceptions\AllowRedirectsMustBeEnabledException
     */
    public function test_construct_should_throw_exception_when_allow_redirects_is_false(): void
    {
        $this->expectException(AllowRedirectsMustBeEnabledException::class);

        $client = $this->createStub(Client::class);
        $client->method('getConfig')->with('allow_redirects')->willReturn(false);

        $factory = new PurchaseAgreementFactory(
            $this->fixturePurchaseAgreement(),
            $client,
            $this->storage,
            'impersonate-key',
            'dealerdirect.eu.aio.test'
        );

        $this->assertTrue($factory->client->getConfig('allow_redirects'));
    }

    public function test_method_create_stores_pdf_in_storage(): void
    {
        $client = $this->createStub(Client::class);
        $client->method('send')->willReturn(
            new Response(200, ['Content-Type' => 'application/pdf'], 'pdf-content')
        );

        $factory = new PurchaseAgreementFactory(
            $this->fixturePurchaseAgreement(),
            $client,
            $this->storage,
            'impersonate-key',
            'dealerdirect.eu.aio.test'
        );

        $factory->create();

        $this->assertTrue(
            $this->storage->exists($factory->getStorageName())
        );
    }

    public function test_method_create_will_not_store_pdf_in_storage_if_already_exists(): void
    {
        $this->expectNotToPerformAssertions();

        $data = $this->fixturePurchaseAgreement();

        foreach (DealerdirectLanguage::cases() as $language) {
            $storageName = $this->generateStorageName($data->company->id, $data->lotId, $data->vehicleType, $language);

            $this->storage->put($storageName, 'pdf-content');
        }

        $client = $this->createStub(Client::class);
        $client->expects($this->never())->method('send');

        $factory = new PurchaseAgreementFactory(
            $data,
            $client,
            $this->storage,
            'impersonate-key',
            'dealerdirect.eu.aio.test'
        );

        $factory->create();
    }

    public function test_method_show_will_return_cached_pdf(): void
    {
        $data = $this->fixturePurchaseAgreement();

        foreach (DealerdirectLanguage::cases() as $language) {
            $storageName = $this->generateStorageName($data->company->id, $data->lotId, $data->vehicleType, $language);

            $this->storage->put($storageName, 'Cached PDF content');
        }

        $factory = new PurchaseAgreementFactory(
            $data,
            $this->createStub(Client::class),
            $this->storage,
            'impersonate-key',
            'dealerdirect.eu.aio.test'
        );

        $this->assertSame('Cached PDF content', $factory->show());
    }

    public function test_method_show_will_return_generated_pdf(): void
    {

        $client = $this->createStub(Client::class);
        $client->method('send')->willReturn(
            new Response(200, ['Content-Type' => 'application/pdf'], 'Generated PDF content')
        );

        $factory = new PurchaseAgreementFactory(
            $this->fixturePurchaseAgreement(),
            $client,
            $this->storage,
            'impersonate-key',
            'dealerdirect.eu.aio.test'
        );

        $factory->create();

        $this->assertSame('Generated PDF content', $factory->show());
    }

    public function test_method_delete_will_remove_pdf_from_storage(): void
    {
        $data = $this->fixturePurchaseAgreement();

        foreach (DealerdirectLanguage::cases() as $language) {
            $storageName = $this->generateStorageName($data->company->id, $data->lotId, $data->vehicleType, $language);

            $this->storage->put($storageName, 'pdf-content');
        }

        $factory = new PurchaseAgreementFactory(
            $data,
            $this->createStub(Client::class),
            $this->storage,
            'impersonate-key',
            'dealerdirect.eu.aio.test'
        );

        $factory->delete();

        $this->assertFalse($this->storage->exists($storageName));
    }

    /**
     * @uses \Dealerdirect\Agreements\Mailers\Mailer
     * @uses \Dealerdirect\Agreements\Mailers\PurchaseAgreementMailer
     * @uses \Dealerdirect\Agreements\Utils\Translator
     */
    public function test_method_send_will_send_email_with_pdf(): void
    {
        $this->expectNotToPerformAssertions();

        $client = $this->createStub(Client::class);
        $client->method('send')->willReturn(
            new Response(200, ['Content-Type' => 'application/pdf'], 'Generated PDF content')
        );

        $factory = new PurchaseAgreementFactory(
            $this->fixturePurchaseAgreement(),
            $client,
            $this->storage,
            'impersonate-key',
            'dealerdirect.eu.aio.test'
        );

        $emailClient = $this->createStub(Email::class);
        $emailClient->expects($this->once())->method('sendEmail');

        $factory->send($emailClient);
    }

    /**
     * @covers \Dealerdirect\Agreements\Exceptions\FetchPurchaseAgreementPdfException
     */
    public function test_method_send_will_throw_exception_when_response_status_code_is_not_200(): void
    {
        $this->expectException(FetchPurchaseAgreementPdfException::class);

        $client = $this->createStub(Client::class);
        $client->method('send')->willReturn(
            new Response(500, ['Content-Type' => 'application/pdf'], 'Internal Server Error')
        );

        $factory = new PurchaseAgreementFactory(
            $this->fixturePurchaseAgreement(),
            $client,
            $this->storage,
            'impersonate-key',
            'dealerdirect.eu.aio.test'
        );

        $factory->send($this->createStub(Email::class));
    }

    /**
     * @covers \Dealerdirect\Agreements\Exceptions\PurchaseAgreementNotStoredException
     */
    public function test_method_store_should_throw_exception_when_storage_fails(): void
    {
        $this->expectException(PurchaseAgreementNotStoredException::class);

        $client = $this->createStub(Client::class);
        $client->method('send')->willReturn(
            new Response(200, ['Content-Type' => 'application/pdf'], 'pdf-content')
        );

        $storage = $this->createStub(FilesystemAdapter::class);
        $storage->method('exists')->willReturn(false);
        $storage->method('put')->willReturn(false);

        $factory = new PurchaseAgreementFactory(
            $this->fixturePurchaseAgreement(),
            $client,
            $storage,
            'impersonate-key',
            'dealerdirect.eu.aio.test'
        );

        $factory->create();
    }

    public function test_request_should_contain_correct_headers(): void
    {
        $client = $this->createStub(Client::class);

        $client->method('send')->with(
            $this->callback(function (Request $request) {

                $this->assertSame('GET', $request->getMethod());
                $this->assertStringContainsString('purchase/motorbike/12345/pdf', (string) $request->getUri());
                $this->assertSame('dealerdirect-agreements', $request->getHeaderLine('X-Requested-By'));
                $this->assertSame('Bearer impersonate-key', $request->getHeaderLine('Authorization'));
                $this->assertSame('motobike', $request->getHeaderLine('Vehicle-Type'));
                $this->assertSame('1', $request->getHeaderLine('BedrijvenID'));

                return true;
            })
        )->willReturn(
            new Response(200, ['Content-Type' => 'application/pdf'], 'pdf-content')
        );

        (new PurchaseAgreementFactory(
            $this->fixturePurchaseAgreement(vehicleType: VehicleType::MOTOR),
            $client,
            $this->storage,
            'impersonate-key',
            'dealerdirect.eu.aio.test'
        ))->create();
    }

    private function generateStorageName(
        int $companyId,
        int $lotId,
        string $vehicleType,
        DealerdirectLanguage $language
    ): string {
        return sprintf(
            'purchase-agreements/%s.pdf',
            sha1($companyId . $lotId . $vehicleType . $language->value)
        );
    }
}
