<?php

declare(strict_types=1);

namespace Dealerdirect\Agreements\Factories;

use Dealerdirect\Agreements\DataObjects;
use Dealerdirect\Agreements\Enums\DealerdirectLanguage;
use Dealerdirect\Agreements\Exceptions\AllowRedirectsMustBeEnabledException;
use Dealerdirect\Agreements\Exceptions\FetchPurchaseAgreementPdfException;
use Dealerdirect\Agreements\Exceptions\PurchaseAgreementNotStoredException;
use Dealerdirect\Agreements\Mailers\PurchaseAgreementMailer;
use Dealerdirect\Agreements\Utils\GeneralUtils;
use DealerDirect\Sdk\Facade\Email;
use GuzzleHttp\Client;
use GuzzleHttp\Psr7\Request;
use Illuminate\Contracts\Filesystem\Filesystem;

final class PurchaseAgreementFactory implements Factory
{
    public const REQUESTED_BY = 'dealerdirect-agreements';

    public function __construct(
        public readonly DataObjects\PurchaseAgreement $data,
        public readonly Client $client,
        private Filesystem $storage,
        private string $impersonateKey,
        private string $dealerdirectUrl,
        private ?DealerdirectLanguage $language = null,
    ) {
        if (false === $this->client->getConfig('allow_redirects')) {
            throw new AllowRedirectsMustBeEnabledException();
        }

        $this->language ??= $this->data->locale->toDealerdirectLanguage();

        // Check if storage has folder, if not create it
        if (!in_array('purchase-agreements', $this->storage->directories())) {
            $this->storage->makeDirectory('purchase-agreements');
        }
    }

    public function setLanguage(DealerdirectLanguage $language): void
    {
        $this->language = $language;
    }

    public function exists(): bool
    {
        return $this->storage->exists(
            $this->getStorageName()
        );
    }

    public function create(): void
    {
        if ($this->exists()) {
            return;
        }

        $currentLanguage = $this->language;

        foreach (DealerdirectLanguage::cases() as $language) {
            $this->setLanguage($language);

            $request = $this->getRequest();
            $pdfContent = $this->fetchPdf($request);

            $this->store($pdfContent);
        }

        $this->language = $currentLanguage;
    }

    public function store(string $pdfContent): void
    {
        $storageName = $this->getStorageName();

        if (!$this->storage->put($storageName, $pdfContent)) {
            throw new PurchaseAgreementNotStoredException(
                $this->data->lotId,
                $this->data->company->id,
                $this->data->siteReference,
                $this->language,
                $storageName
            );
        }
    }

    public function show(): string
    {
        if (!$this->exists()) {
            $this->create();
        }

        return $this->storage->get(
            $this->getStorageName()
        );
    }

    public function delete(): void
    {
        $currentLanguage = $this->language;

        foreach (DealerdirectLanguage::cases() as $language) {
            $this->setLanguage($language);

            $storageName = $this->getStorageName();
            if ($this->storage->exists($storageName)) {
                $this->storage->delete($storageName);
            }
        }

        $this->language = $currentLanguage;
    }

    /**
     * @param array<string,string> $domains
     */
    public function send(Email $emailClient, ?array $domains = null, ?string $encryptionKey = null): void
    {
        PurchaseAgreementMailer::create($emailClient, $this->data, $this->show())->send();
    }

    public function getStorageName(): string
    {
        return sprintf('purchase-agreements/%s.pdf', sha1(
            $this->data->company->id
            . $this->data->lotId
            . $this->data->vehicleType
            . $this->language->value
        ));
    }

    private function getRequest(?DealerdirectLanguage $language = null): Request
    {
        $language ??= $this->language;

        return new Request('GET', $this->getPdfUrl($language), [
            'Accept-Language' => $language->value,
            'Authorization' => "Bearer {$this->impersonateKey}",
            'BedrijvenID' => (int) $this->data->company->id,
            'Vehicle-Type' => $this->getDealerdirectVehicleType(),
            'X-Requested-By' => self::REQUESTED_BY,
        ]);
    }

    private function getDealerdirectVehicleType(): string
    {
        return GeneralUtils::isMotor($this->data->vehicleType)
            ? 'motobike'
            : $this->data->vehicleType;
    }

    private function getPdfUrl(DealerdirectLanguage $language): string
    {
        return sprintf(
            '%s/%s/purchase/%s/%d/pdf',
            $this->dealerdirectUrl,
            $language->value,
            $this->data->vehicleType,
            $this->data->lotId,
        );
    }

    private function fetchPdf(Request $request): string
    {
        $response = $this->client->send($request);
        if (200 !== $response->getStatusCode()) {
            throw new FetchPurchaseAgreementPdfException(
                $this->data->lotId,
                $this->data->company->id,
                $this->data->locale->value,
                $this->data->vehicleType,
                $response->getStatusCode(),
                $response->getHeaderLine('Content-Type')
            );
        }

        if ('application/pdf' !== $response->getHeaderLine('Content-Type')) {
            throw new FetchPurchaseAgreementPdfException(
                $this->data->lotId,
                $this->data->company->id,
                $this->data->locale->value,
                $this->data->vehicleType,
                $response->getStatusCode(),
                $response->getHeaderLine('Content-Type'),
            );
        }

        return (string) $response->getBody();
    }
}
