<?php

namespace Dealerdirect\Agreements;

use League\Flysystem\Filesystem;

class Storage
{
    ////////////////////////////// CLASS PROPERTIES \\\\\\\\\\\\\\\\\\\\\\\\\\\\

    public const DEFAULT_CONTENT_TYPE = 'application/octet';

    public const ERROR_ACTION_FAILED = 'Could not %s path "%s" %s';
    public const ERROR_ENV_NOT_SET = 'The location to store sales agreements has not been set in the environment';
    public const ERROR_FILESYSTEM_MISSING = 'Filesystem is not available';
    public const ERROR_FILESYSTEM_NOT_CREATED = 'Could not create file-system for "%s" ';

    /**
     * @var Filesystem
     */
    private $filesystem;
    /**
     * @var self
     */
    private static $instance;

    //////////////////////////// SETTERS AND GETTERS \\\\\\\\\\\\\\\\\\\\\\\\\\\

    /**
     * @return Filesystem
     *
     * @throws \UnexpectedValueException
     */
    final public function getFileSystem()
    {
        if ($this->filesystem === null) {
            throw new \UnexpectedValueException(self::ERROR_FILESYSTEM_MISSING);
        }

        return $this->filesystem;
    }

    /**
     * @return SalesAgreementStorage
     */
    final public static function getInstance($location)
    {
        if (self::$instance === null) {
            $error = '';
            $filesystem = null;

            if ($location === false) {
                $error = self::ERROR_ENV_NOT_SET;
            } else {
                try {
                    $filesystem = \MJRider\FlysystemFactory\create($location);
                } catch (\Exception $exception) {
                }

                if ($filesystem === null) {
                    $error = sprintf(self::ERROR_FILESYSTEM_NOT_CREATED, $location);

                    if (isset($exception)) {
                        $error = $exception->getMessage();
                    }
                }
            }

            $instance = new self($filesystem);

            $instance->logError($error);

            self::$instance = $instance;
        }

        return self::$instance;
    }

    //////////////////////////////// PUBLIC API \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

    /**
     * @CHECKME: Should the constructor be private or not?  2017/11/30/BMP
     *
     * @param Filesystem|null $filesystem
     */
    final public function __construct(Filesystem $filesystem = null)
    {
        $this->filesystem = $filesystem;
    }

    final public static function getContentType($location, $path)
    {
        $contentType = self::getInstance($location)->call('getMimetype', [$path]);

        if ($contentType === false) {
            $contentType = self::DEFAULT_CONTENT_TYPE;
        }

        return $contentType;
    }

    /**
     * @param string $path
     *
     * @return bool
     *
     * @throws \UnexpectedValueException
     */
    final public static function has($location, $path)
    {
        return self::getInstance($location)->call(__FUNCTION__, [$path]);
    }

    /**
     * @param string $path
     * @param string $content
     *
     * @return bool
     *
     * @throws \UnexpectedValueException
     */
    final public static function put($location, $path, $content)
    {
        return self::getInstance($location)->call(__FUNCTION__, [$path, $content]);
    }

    /**
     * Create a file or update if exists.
     *
     * @param string   $path     The path to the file.
     * @param resource $resource The file handle.
     *
     * @return bool True on success, false on failure.
     *
     * @throws \InvalidArgumentException Thrown if $resource is not a resource.
     */
    final public static function putStream($location, $path, $resource)
    {
        return self::getInstance($location)->call(__FUNCTION__, [$path, $resource]);
    }

    /**
     * @param string $path
     *
     * @return string
     *
     * @throws \UnexpectedValueException
     */
    final public static function read($location, $path)
    {
        return self::getInstance($location)->call(__FUNCTION__, [$path]);
    }

    /**
     * @param string $path
     *
     * @return resource
     *
     * @throws \UnexpectedValueException
     */
    final public static function readStream($location, $path)
    {
        return self::getInstance($location)->call(__FUNCTION__, [$path]);
    }

    ////////////////////////////// UTILITY METHODS \\\\\\\\\\\\\\\\\\\\\\\\\\\\\

    /**
     * @param string $method
     * @param array  $params
     *
     * @return mixed
     *
     * @throws \UnexpectedValueException
     */
    private function call($method, array $params)
    {
        $result = null;

        $filesystem = $this->getFileSystem();

        try {
            $result = call_user_func_array([$filesystem, $method], $params);
        } catch (\Exception $exception) {
            /* @NOTE: The first parameter of Flysystem methods is always a path */
            $path = reset($params);
            $error = sprintf(
                self::ERROR_ACTION_FAILED,
                $method,
                $path,
                $exception->getMessage()
            );
            $this->logError($error);
        }

        return $result;
    }

    /**
     * @param string $error
     *
     * @return bool
     */
    private function logError($error)
    {
        $result = true;

        if ($error !== '') {
            $result = error_log($error);
        }

        return $result;
    }
}
