<?php

namespace Dealerdirect\Agreements\Agreements;

use Dealerdirect\Agreements\Interfaces\Agreement;
use Dealerdirect\Agreements\Interfaces\Model;
use DealerDirect\Generic\Category\Locale as LocaleCat;
use DealerDirect\Generic\Category\TranslationTypes;
use DealerDirect\Generic\Category\VehicleType;
use DealerDirect\Generic\Repository\GenericProperties;
use DealerDirect\Generic\Repository\Locale;
use DealerDirect\Generic\Repository\Translator;
use DealerDirect\Sdk\Config;
use DealerDirect\Sdk\Facade\Email;
use Dealerdirect\Sdk\Model\Email\Email as EmailModel;
use DealerDirect\Sdk\Sdk;

abstract class AbstractAgreement implements Agreement
{
    /**
     * @var array $data
     */
    protected $data;
    /**
     * @var Model $model
     */
    protected $model;
    /**
     * @var Translator $translator
     */
    protected $translator;
    /**
     * @var GenericProperties $genericProperties
     */
    protected $genericProperties;
    /**
     * @var string $emailApiUrl
     */
    private $emailApiUrl;

    /**
     * Creates the agreement
     *
     * @return boolean
     */
    abstract public function create(): bool;

    /**
     * Sends the agreement to email-service
     *
     * @return boolean
     */
    abstract public function send(): bool;

    /**
     * AbstractAgreement constructor
     *
     * @param Model  $model
     * @param string $emailApiUrl
     */
    public function __construct(Model $model, string $emailApiUrl)
    {
        $this->model = $model;
        $this->emailApiUrl = $emailApiUrl;
    }

    /**
     * Set's some default data, called inside the __constructor
     *
     * @param  Model $model
     * @return void
     */
    protected function setDefaultData(Model $model): void
    {
        $vehicleType = $model->getVehicleType();
        if (in_array($vehicleType, [VehicleType::SCOOTER, VehicleType::BICYCLE])) {
            $vehicleType = VehicleType::MOTOR;
        }
        $this->genericProperties = new GenericProperties(
            Locale::fromString($model->getLocale()),
            $vehicleType
        );
        $this->translator = new Translator($this->genericProperties);
    }

    /**
     * Sends the email to email-service
     *
     * @param  EmailModel $email
     * @param  array      $to
     * @param  array      $cc
     * @param  array      $bcc
     * @return bool
     */
    protected function sendEmail(EmailModel $email, array $to, array $cc = [], array $bcc = []): bool
    {
        try {
            if (count($to) > 0) {
                $emailClient = $this->createEmailClient();
                $emailClient->sendEmail($email, $to, $cc, $bcc);
            }
            $result = true;
        } catch (\Exception $exception) {
            $result = false;
            if (extension_loaded('newrelic')) {
                newrelic_notice_error(null, $exception);
            }
            error_log(
                sprintf(
                    "Exception when sending email %s [%s] with message: %s",
                    print_r($email, true),
                    print_r($to, true),
                    $exception->getMessage()
                ),
                4
            );
        }

        return $result;
    }

    /**
     * Gets translated country name from countryId
     *
     * @param  int    $countryId
     * @param  string $default
     * @return mixed
     */
    protected function countryIdToTranslated(int $countryId, $default = '-')
    {
        $countries = $this->translator->getTranslations(TranslationTypes::GENERAL_VALUES, 'country');

        return $countries[$countryId] ?? $default;
    }

    /**
     * Gets the address for consumer / company
     * Function called by getConsumerAddress or getCompanyAddress
     *
     * @param  string $type
     * @return string
     */
    private function getAddress(string $type): string
    {
        return sprintf(
            "%s %s\n%s %s\n%s",
            $this->data["{$type}_street_name"],
            $this->data["{$type}_house_number"],
            $this->model->getLocale() === LocaleCat::NL_NL && $type === 'consumer'
                ? $this->data["{$type}_zip_code_nl"]
                : $this->data["{$type}_zip_code"],
            $this->data["{$type}_city"],
            $this->countryIdToTranslated($this->data["{$type}_country"])
        );
    }

    /**
     * Gets the company address from data.
     *
     * @return string
     */
    protected function getCompanyAddress(): string
    {
        return $this->getAddress('company');
    }

    /**
     * gets the consumer address from data
     *
     * @return string
     */
    protected function getConsumerAddress(): string
    {
        return $this->getAddress('consumer');
    }

    /**
     * Creates the email client with the correct config
     *
     * @param  string $emailApiUrl
     * @return Email
     */
    protected function createEmailClient(): Email
    {
        $config = new Config([
            Config::EMAIL_SERVICE_URL => $this->emailApiUrl,
        ]);

        return (new Sdk($config))->getEmail();
    }
}
