<?php

namespace Dealerdirect\Deliveries;

use Dealerdirect\Deliveries\Config;
use Dealerdirect\Deliveries\Exceptions as Exceptions;
use Dealerdirect\Deliveries\Repositories as Repositories;
use Doctrine\DBAL\Connection;

class Delivery
{
    /** @var AbstractRepository $repository */
    private $repository;

    /** @var array $data */
    private $data = [];

    /**
     * @param Connection $connection
     * @param Config $config
     * @param string $country
     */
    public function __construct(Repositories\AbstractRepository $repository)
    {
        $this->repository = $repository;
    }

    /**
     * Constructs Delivery class with vehicleType and countryCode
     *
     * @param string $vehicleType
     * @param string $countryCode
     * @param Connection $connection
     * @return self
     */
    public static function constructWithVehicleTypeAndCountry(
        string $vehicleType,
        string $countryCode,
        Connection $connection
    ): self {
        $branch = strtolower("{$vehicleType}_{$countryCode}");
        $repositories = [
            'car_nl' => Repositories\CarNlRepository::class,
            'car_be' => Repositories\CarBeRepository::class,
            'car_de' => Repositories\CarDeRepository::class,
            'motorbike_nl' => Repositories\MotorNlRepository::class,
            'motorbike_be' => Repositories\MotorBeRepository::class,
            'motorbike_de' => Repositories\MotorDeRepository::class,
            'scooter_nl' => Repositories\ScooterNlRepository::class,
            'bicycle_nl' => Repositories\BicycleNlRepository::class,
        ];

        if (!array_key_exists($branch, $repositories)) {
            throw Exceptions\DeliveryException::invalidBranch($branch);
        }

        return new self(new $repositories[$branch]($connection));
    }

    /**
     * @param integer $bidId
     * @return array
     */
    public function getSaleData(int $bidId): array
    {
        if (empty($this->data)) {
            $this->data = $this->repository->getSaleData($bidId);
        }

        return $this->data;
    }

    /**
     * @param integer $bidId
     * @param integer $agentId
     * @param string $ipAddress
     * @param DateTime $dateOfBirth
     * @throws Exceptions\AddSaleException
     * @return int
     */
    public function addSale(int $bidId, int $agentId, string $ipAddress, \DateTime $dateOfBirth): int
    {
        $this->repository->beginTransaction();
        try {
            $data = $this->getSaleData($bidId);
            $this->validate($data);

            // Pull lotId and remove make, model, reference from $data as its not needed for insert.
            $lotId = $data['lotId'];
            unset($data['lotId'], $data['make'], $data['model'], $data['reference']);

            // Merge data for inserts
            $data = array_merge(
                $data,
                ['verkoper' => $agentId, 'ip' => $ipAddress, 'afhandeling_mail' => 1, 'geboortedatum' => $dateOfBirth]
            );

            if ($this->repository->updateStatusToSold($lotId) === false) {
                throw Exceptions\AddSaleException::updateLotStatus($lotId);
            } elseif ($this->repository->updateLockedSale($lotId) === false) {
                throw Exceptions\AddSaleException::updateLockedSale($lotId);
            }

            $deliverId = $this->repository->create($data);
            $this->repository->setContactDate(
                $deliverId,
                $this->repository->getCountryId(),
                $this->calculateContactDate($agentId, (int) $data['bem_bedrag'])
            );
            $this->repository->commit();
        } catch (\Exception $exception) {
            $this->repository->rollBack();

            throw $exception;
        }

        return $deliverId;
    }

    /**
     * Updates column agreements_created
     *
     * @param integer $deliverId
     * @param boolean $value
     * @return bool
     */
    public function updateAgreementsCreated(int $deliverId, bool $value): bool
    {
        return $this->repository->updateAgreementsCreated($deliverId, $value);
    }

    private function validate(array $data)
    {
        $requiredKeys = [
            'lotId', 'kavelref', 'reference', 'aan_aanhef_kl', 'aan_naam_kl', 'aan_straat_kl', 'aan_huisnummer_kl',
            'aan_postcode_kl', 'aan_woonplaats_kl', 'aan_land_kl', 'aan_tel_kl', 'aan_gsm_kl', 'aan_provincie_kl',
            'aan_email_kl', 'pakket', 'bonus_profile', 'bid_id', 'verkoop_bedrag', 'bem_bedrag', 'gehaald',
            'bedrijvenID', 'make', 'model',
        ];

        $errors = [];
        foreach ($requiredKeys as $requiredKey) {
            if (!array_key_exists($requiredKey, $data)) {
                $errors[] = $requiredKey;
            }
        }

        if (!empty($errors)) {
            throw Exceptions\AddSaleException::invalidData($errors);
        }
    }

    /**
     * @param integer $agentId
     * @param integer $fee
     * @return \DateTime
     */
    private function calculateContactDate(int $agentId, int $fee): \DateTime
    {
        return new \DateTime('today + 5 day');
    }
}
