<?php

namespace Dealerdirect\Deliveries;

use Dealerdirect\Agreements\Agreements\PurchaseAgreement;
use Dealerdirect\Agreements\Agreements\SalesAgreement;
use Dealerdirect\Agreements\Interfaces\Agreement;
use Dealerdirect\Agreements\Interfaces\Model;
use Dealerdirect\Agreements\Models as Models;
use Dealerdirect\Deliveries\Config;
use Dealerdirect\Deliveries\Exceptions as Exceptions;
use Dealerdirect\Deliveries\Repositories as Repositories;
use Dealerdirect\SMS as SMS;
use Doctrine\DBAL\Connection;

class Delivery
{
    /** @var AbstractRepository $repository */
    private $repository;

    /** @var Config $config */
    private $config;

    /** @var array $data */
    private $data = [];

    /** @var Models\AbstractModel $model */
    private $model;

    /**
     * @param Connection $connection
     * @param Config $config
     * @param string $country
     */
    public function __construct(Repositories\AbstractRepository $repository, Config $config)
    {
        $this->config = $config;
        $this->repository = $repository;
    }

    /**
     * Constructs Delivery class with vehicleType and countryCode
     *
     * @param string $vehicleType
     * @param string $countryCode
     * @param Config $config
     * @param Connection $connection
     * @return self
     */
    public static function constructWithVehicleTypeAndCountry(
        string $vehicleType,
        string $countryCode,
        Config $config,
        Connection $connection
    ): self {
        $branch = strtolower("{$vehicleType}_{$countryCode}");
        switch ($branch) {
            case 'car_nl':
                $repository = new Repositories\CarNlRepository($connection);
                break;
            case 'car_be':
                $repository = new Repositories\CarBeRepository($connection);
                break;
            case 'car_de':
                $repository = new Repositories\CarDeRepository($connection);
                break;
            case 'motorbike_nl':
            case 'motorbike_be':
            case 'motorbike_de':
            case 'scooter_nl':
                $repository = new Repositories\MotorRepository($connection);
                break;
            default:
                throw Exceptions\DeliveryException::invalidBranch($branch);
                break;
        }

        return new self($repository, $config);
    }

    /**
     * @param integer $bidId
     * @return array
     */
    public function getSaleData(int $bidId): array
    {
        if (empty($this->data)) {
            $this->data = $this->repository->getSaleData($bidId);
        }

        return $this->data;
    }

    /**
     * @param integer $bidId
     * @param integer $agentId
     * @param string $ipAddress
     * @param DateTime $dateOfBirth
     * @throws Exceptions\AddSaleException
     * @return int
     */
    public function addSale(int $bidId, int $agentId, string $ipAddress, \DateTime $dateOfBirth): int
    {
        $this->repository->beginTransaction();
        try {
            $data = $this->getSaleData($bidId);
            $this->validate($data);

            // Pull lotId and remove make, model, reference from $data as its not needed for insert.
            $lotId = $data['lotId'];
            unset($data['lotId'], $data['make'], $data['model'], $data['reference']);

            // Merge data for inserts
            $data = array_merge(
                $data,
                ['verkoper' => $agentId, 'ip' => $ipAddress, 'afhandeling_mail' => 1, 'geboortedatum' => $dateOfBirth]
            );

            if ($this->repository->updateStatusToSold($lotId) === false) {
                throw Exceptions\AddSaleException::updateLotStatus($lotId);
            } elseif ($this->repository->updateLockedSale($lotId) === false) {
                throw Exceptions\AddSaleException::updateLockedSale($lotId);
            }

            $deliverId = $this->repository->create($data);
            $this->repository->setContactDate(
                $deliverId,
                (int) $data['aan_land_kl'],
                $this->calculateContactDate($agentId, (int) $data['bem_bedrag'])
            );
            $this->repository->commit();
        } catch (\Exception $exception) {
            $this->repository->rollBack();

            throw $exception;
        }

        return $deliverId;
    }

    /**
     * @param integer $deliverId
     * @param string $ipAddress
     * @return boolean
     */
    public function sendAgreements(int $deliverId, string $ipAddress): bool
    {
        $agreementsModel = $this->getAgreementsModel($deliverId);
        foreach ($this->getAgreements($agreementsModel) as $agreement) {
            try {
                $this->sendAgreement($agreement, $agreementsModel, $ipAddress);
            } catch (\Exception $exception) {
                $exceptions[] = $exception;
            }
        }

        if (!empty($exceptions)) {
            throw Exceptions\DeliveryException::sendAgreements($exceptions);
        }

        return true;
    }

    /**
     * @param string $vehicleType
     * @param string $countryCode
     * @param integer $deliverId
     * @param string $ipAddress
     * @return void
     */
    public function sendAgreeSaleSms(string $vehicleType, string $countryCode, int $deliverId, string $ipAddress): void
    {
        $data = $this->getSmsData($deliverId);
        $sms = new SMS\SMS(
            SMS\Helper::getClient(
                $this->config->get(Config::VONAGE_API_KEY),
                $this->config->get(Config::VONAGE_API_SECRET)
            ),
            new SMS\Messages\AgreeSale($vehicleType, strtoupper($countryCode), [
                'make' => $data['make'],
                'model' => $data['model'],
                'emailAddress' => $data['email'],
            ])
        );

        $phoneNumber = $sms->send([$data['phone1'], $data['phone2']]);
        if ($phoneNumber) {
            $model = $this->model ?: $this->getAgreementsModel($deliverId);
            // TODO: Add IP address ?
            $model->log("Verkoop SMS: {$phoneNumber}", 0, $ipAddress);
        }
    }

    private function getSmsData(int $deliverId)
    {
        if (!empty($this->data)) {
            return [
                'phone1' => $this->data['aan_tel_kl'],
                'phone2' => $this->data['aan_gsm_kl'],
                'email' => $this->data['aan_email_kl'],
                'make' => $this->data['make'],
                'model' => $this->data['model'],
            ];
        }

        return $this->repository->getSmsData($deliverId);
    }

    private function validate(array $data)
    {
        $requiredKeys = [
            'lotId', 'kavelref', 'reference', 'aan_aanhef_kl', 'aan_naam_kl', 'aan_straat_kl', 'aan_huisnummer_kl',
            'aan_postcode_kl', 'aan_woonplaats_kl', 'aan_land_kl', 'aan_tel_kl', 'aan_gsm_kl', 'aan_provincie_kl',
            'aan_email_kl', 'pakket', 'bonus_profile', 'bid_id', 'verkoop_bedrag', 'bem_bedrag', 'gehaald',
            'bedrijvenID', 'make', 'model',
        ];

        $errors = [];
        foreach ($requiredKeys as $requiredKey) {
            if (!array_key_exists($requiredKey, $data)) {
                $errors[] = $requiredKey;
            }
        }

        if (!empty($errors)) {
            throw Exceptions\AddSaleException::invalidData($errors);
        }
    }

    /**
     * @param Model $model
     * @return array
     */
    private function getAgreements(Model $model)
    {
        return [
            'salesAgreement' => new SalesAgreement(
                $model,
                $this->config->get(Config::EMAIL_SERVICE_API_URL),
                $this->config->get(Config::SALES_AGREEMENT_LOCATION)
            ),
            'purchaseAgreement' => new PurchaseAgreement(
                $model,
                $this->config->get(Config::EMAIL_SERVICE_API_URL),
                $this->config->get(Config::DEALER_SITE_IMPERSONATE_KEY),
                $this->config->get(Config::DEALER_SITE_URL)
            ),
        ];
    }

    /**
     * @param Agreement $agreement
     * @param Model $model
     * @return boolean
     */
    private function sendAgreement(Agreement $agreement, Model $model, string $ipAddress): bool
    {
        $success = $agreement->create() && $agreement->send();
        $log = sprintf(
            '%s %s',
            $agreement instanceof SalesAgreement ? 'VERKOOPOVEREENKOMST' : 'AKKOORD AANKOOP',
            $success ? 'Verstuurd' : 'NIET Verstuurd'
        );
        $model->log($log, 0, $ipAddress);

        return $success;
    }

    /**
     * @param integer $deliverId
     * @return Models\Model
     * @throws \Exception
     */
    private function getAgreementsModel(int $deliverId): Models\AbstractModel
    {
        if ($this->repository instanceof Repositories\CarBeRepository) {
            $this->model = new Models\CarBeModel($this->repository->connection, $deliverId);
        } elseif ($this->repository instanceof Repositories\CarNlRepository) {
            $this->model = new Models\CarNlModel($this->repository->connection, $deliverId);
        } elseif ($this->repository instanceof Repositories\MotorRepository) {
            $this->model = new Models\MotorbikeModel($this->repository->connection, $deliverId);
        } else {
            throw new \Exception('Could not get agreements Model.');
        }

        return $this->model;
    }

    /**
     * @param integer $agentId
     * @param integer $fee
     * @return \DateTime
     */
    private function calculateContactDate(int $agentId, int $fee): \DateTime
    {
        if ($agentId === 0) {
            return new \DateTime('today');
        } elseif ($fee >= 500) {
            return new \DateTime('today + 1 day');
        } elseif ($fee >= 100) {
            return new \DateTime('today + 2 day');
        }

        return new \DateTime('today + 3 day');
    }
}
