<?php

declare(strict_types=1);

namespace Dealerdirect\Deliveries;

use DateTime;
use PHPUnit\Framework\TestCase;
use Dealerdirect\Deliveries\Config;
use Dealerdirect\Deliveries\Delivery;
use Dealerdirect\Deliveries\Exceptions\AddSaleException;
use Dealerdirect\Deliveries\Exceptions\DeliveryException;
use Dealerdirect\Deliveries\Repositories\CarNlRepository;
use Dealerdirect\Deliveries\Exceptions\RepositoryException;
use DealerDirect\Generic\Category\VehicleType;
use Doctrine\DBAL\Connection;

class DeliveryTest extends TestCase
{
    public function testConstructWithVehicleTypeAndCountryCodeShouldThrowExceptionWhenInvalidVehicleTypeOrCountryCode()
    {
        $this->expectException(DeliveryException::class);
        $this->expectExceptionMessage("Invalid branch 'vehicletype_country'.");

        $config = $this->fixtureGetValidConfigClass();
        /** @var Connection $connection */
        $connection = $this->createStub(Connection::class);

        Delivery::constructWithVehicleTypeAndCountry('vehicletype', 'country', $config, $connection);
    }

    /**
     * @dataProvider provideValidVehicleTypesAndCountryCodes
     */
    public function testConstructWithVehicleTypeAndCountryShouldReturnNewInstance($vehicleType, $countryCode)
    {
        $expectedInstance = Delivery::class;

        $config = $this->fixtureGetValidConfigClass();
        /** @var Connection $connection */
        $connection = $this->createStub(Connection::class);

        $delivery = Delivery::constructWithVehicleTypeAndCountry($vehicleType, $countryCode, $config, $connection);
        $this->assertTrue($delivery instanceof $expectedInstance);
    }

    /**
     * @covers Delivery::addSale || RepositoryException
     */
    public function testAddSaleShouldThrowAddSaleExceptionWhenEmptyData()
    {
        $this->expectException(RepositoryException::class);
        $this->expectExceptionMessageMatches('/Empty data with \[Query: ".*?"; Params: bidId: 123456\]/');

        $connection = $this->createStub(Connection::class);
        $connection->method('fetchAssociative')->willReturn(false);
        $connection->method('beginTransaction')->willReturn(true);
        $connection->method('rollback')->willReturn(true);
        /** @var Connection $connection */

        $repository = new CarNlRepository($connection);

        $delivery = new Delivery($repository, $this->fixtureGetValidConfigClass(), 'car');
        $delivery->addSale(123456, 1234, '192.168.1.1', new DateTime('14-03-1991'));
    }

    /**
     * @covers Delivery::addSale || DeliveryException
     */
    public function testAddSaleShouldThrowAddSaleExceptionWhenDataMissesRequiredKeys()
    {
        $this->expectException(DeliveryException::class);
        $missingKeys = ['verkoop_bedrag', 'bem_bedrag'];
        foreach ($missingKeys as $missingKey) {
            $this->expectExceptionMessageMatches("/{$missingKey}/");
        }


        $repository = $this->createStub(CarNlRepository::class);
        $saleData = $this->fixtureGetValidSaleData();
        unset($saleData['verkoop_bedrag'], $saleData['bem_bedrag']);
        $repository->method('getSaleData')->willReturn($saleData);

        /** @var CarNlRepository $repository */
        $delivery = new Delivery($repository, $this->fixtureGetValidConfigClass(), 'car');
        $delivery->addSale(123456, 1234, '192.168.1.1', new DateTime('14-03-1991'));
    }

    /**
     * @covers Delivery::updateStatusToSold || AddSaleException
     */
    public function testUpdateStatusToSoldShouldRollbackOnErrorAndThrowException()
    {
        $this->expectException(AddSaleException::class);
        $this->expectExceptionMessage('Lot status not updated to sold with [lotId: 8836102].');

        $repository = $this->createStub(CarNlRepository::class);
        $repository->method('getSaleData')->willReturn($this->fixtureGetValidSaleData());
        $repository->method('updateStatusToSold')->willReturn(false);
        $repository->expects($this->once())->method('rollback');

        /** @var CarNlRepository $repository */
        $delivery = new Delivery($repository, $this->fixtureGetValidConfigClass(), 'car');
        $delivery->addSale(123456, 1234, '192.168.1.1', new DateTime('14-03-1991'));
    }

    /**
     * @covers Delivery::updateLockedSale || AddSaleException
     */
    public function testUpdateLockedSaleShouldRollbackOnErrorAndThrowException()
    {
        $this->expectException(AddSaleException::class);
        $this->expectExceptionMessage('Could not update locked sale with [lotId: 8836102]');

        $repository = $this->createStub(CarNlRepository::class);
        $repository->method('getSaleData')->willReturn($this->fixtureGetValidSaleData());
        $repository->method('updateStatusToSold')->willReturn(true);
        $repository->method('updateLockedSale')->willReturn(false);
        $repository->expects($this->once())->method('rollback');

        /** @var CarNlRepository $repository */
        $delivery = new Delivery($repository, $this->fixtureGetValidConfigClass(), 'car');
        $delivery->addSale(123456, 1234, '192.168.1.1', new DateTime('14-03-1991'));
    }

    /**
     * @covers Delivery::updateLockedSale || AddSaleException
     */
    public function testCreateShouldRollbackOnErrorAndThrowException()
    {
        $this->expectException(RepositoryException::class);
        // $this->expectExceptionMessage('Could not update locked sale with [lotId: 8836102]');

        $repository = $this->createStub(CarNlRepository::class);
        $repository->method('getSaleData')->willReturn($this->fixtureGetValidSaleData());
        $repository->method('updateStatusToSold')->willReturn(true);
        $repository->method('updateLockedSale')->willReturn(true);

        $withData = $this->fixtureGetValidSaleData();
        unset($withData['make'], $withData['model'], $withData['reference'], $withData['lotId']);
        $withData = array_merge($withData, [
            'verkoper' => 1234,
            'ip' => '192.168.1.1',
            'afhandeling_mail' => 1,
            'geboortedatum' => new DateTime('14-03-1991'),
        ]);

        $repository->method('create')->with($withData)->willThrowException(
            RepositoryException::insert('dealer01_ddmain.verkopen_levering', $withData)
        );

        $repository->expects($this->once())->method('rollback');

        /** @var CarNlRepository $repository */
        $delivery = new Delivery($repository, $this->fixtureGetValidConfigClass(), 'car');
        $delivery->addSale(123456, 1234, '192.168.1.1', new DateTime('14-03-1991'));
    }

    /**
     * @dataProvider provideContactDateValues
     * @covers Delivery::calculateContactDate
     */
    public function testSetContactDateShouldBeCalledWithCalculatedDatetime($expected, $agentId, $saleData)
    {
        $repository = $this->createStub(CarNlRepository::class);
        $repository->method('getSaleData')->willReturn($saleData);
        $repository->method('updateStatusToSold')->willReturn(true);
        $repository->method('updateLockedSale')->willReturn(true);
        $repository->method('create')->willReturn(12345);
        $repository->expects($this->once())->method('setContactDate')->with(12345, 1, $expected);

        /** @var CarNlRepository $repository */
        $delivery = new Delivery($repository, $this->fixtureGetValidConfigClass(), 'car');
        $delivery->addSale(123456, $agentId, '192.168.1.1', new DateTime('14-03-1991'));
    }

    /**
     * @covers Delivery::addSale || AbstractRepository::setContactDate
     */
    public function testAddSaleShouldRollbackAndThrowExceptionWhensetContactDateFailed()
    {
        $this->expectException(RepositoryException::class);

        $repository = $this->createStub(CarNlRepository::class);
        $repository->method('getSaleData')->willReturn($this->fixtureGetValidSaleData());
        $repository->method('updateStatusToSold')->willReturn(true);
        $repository->method('updateLockedSale')->willReturn(true);
        $repository->method('create')->willReturn(12345);

        $table = 'dealer01_ddmain.verkopen_levering_additional';
        $inserts = [
            'verkopenlevering_id' => 12345,
            'country_id' => 1,
            'date_contact' => new DateTime('today'),
        ];
        $repository->method('setContactDate')->willThrowException(RepositoryException::insert($table, $inserts));
        $repository->expects($this->once())->method('rollback');

        /** @var CarNlRepository $repository */
        $delivery = new Delivery($repository, $this->fixtureGetValidConfigClass(), 'car');
        $delivery->addSale(123456, 0, '192.168.1.1', new DateTime('14-03-1991'));
    }

    /**
     * @covers Delivery::addSale || AbstractRepository::setContactDate
     */
    public function testAddSaleShouldRollbackAndThrowExceptionWhenCommitFailed()
    {
        $this->expectException(RepositoryException::class);
        $this->expectExceptionMessage('Add sale failed! Could not commit changes');

        $repository = $this->createStub(CarNlRepository::class);
        $repository->method('getSaleData')->willReturn($this->fixtureGetValidSaleData());
        $repository->method('updateStatusToSold')->willReturn(true);
        $repository->method('updateLockedSale')->willReturn(true);
        $repository->method('create')->willReturn(12345);
        $repository->method('setContactDate')->willReturn(true);
        $repository->method('commit')->willThrowException(RepositoryException::commit());

        $repository->expects($this->once())->method('rollback');

        /** @var CarNlRepository $repository */
        $delivery = new Delivery($repository, $this->fixtureGetValidConfigClass(), 'car');
        $delivery->addSale(123456, 0, '192.168.1.1', new DateTime('14-03-1991'));
    }

    /**
     * @covers Delivery::addSale
     */
    public function testAddSaleShouldReturnNewDeliverIdOnSuccess()
    {
        $expected = 12345;

        $repository = $this->createStub(CarNlRepository::class);
        $repository->method('getSaleData')->willReturn($this->fixtureGetValidSaleData());
        $repository->method('updateStatusToSold')->willReturn(true);
        $repository->method('updateLockedSale')->willReturn(true);
        $repository->method('create')->willReturn(12345);
        $repository->method('setContactDate')->willReturn(true);
        $repository->method('commit')->willReturn(true);

        /** @var CarNlRepository $repository */
        $delivery = new Delivery($repository, $this->fixtureGetValidConfigClass(), 'car');
        $actual = $delivery->addSale(123456, 0, '192.168.1.1', new DateTime('14-03-1991'));

        $this->assertEquals($expected, $actual);
    }

    public function testSendAgreementsShouldThrowExceptionWhenOneOfThemFailsSending()
    {
        $this->expectException(DeliveryException::class);
        $this->expectExceptionMessageMatches("/Could not send agreement\(s\); Error messages: (.*?)/");

        $connection = $this->createStub(Connection::class);
        $connection->method('fetchAssociative')->willReturn([
            'sales_price' => 10,
            'sales_fee' => 10,
            'company_id' => 123456,
            'sales_date' => '2021-01-01',
            'make' => 'BMW',
            'model' => '3-SERIE',
            'lot_id' => 1234567,
        ]);
        $repository = $this->createStub(CarNlRepository::class);
        $repository->connection = $connection;

        /** @var CarNlRepository $repository */
        $config = $this->fixtureGetValidConfigClass();

        $delivery = new Delivery($repository, $config);
        $delivery->sendAgreements(123456, '192.168.1.1');
    }

    public function provideValidVehicleTypesAndCountryCodes()
    {
        return [
            'car_nl' => [VehicleType::CAR, 'NL'],
            'car_be' => [VehicleType::CAR, 'BE'],
            'car_de' => [VehicleType::CAR, 'DE'],
            'motorbike_nl' => [VehicleType::MOTOR, 'NL'],
            'motorbike_be' => [VehicleType::MOTOR, 'BE'],
            'motorbike_de' => [VehicleType::MOTOR, 'DE'],
            'scooter_nl' => [VehicleType::SCOOTER, 'NL'],
        ];
    }

    public function provideContactDateValues()
    {
        $saleData = $this->fixtureGetValidSaleData();

        return [
            'internet sale and high fee (agentId = 0)' => [
                new DateTime('today'), 0, array_merge($saleData, ['bem_bedrag' => 1200]),
            ],
            'internet sale and low fee (agentId = 0)' => [
                new DateTime('today'), 0, array_merge($saleData, ['bem_bedrag' => 50]),
            ],
            'agent sale and fee > 100 (agentId != 0)' => [
                new DateTime('today + 2 days'), 1, array_merge($saleData, ['bem_bedrag' => 120]),
            ],
            'agent sale and fee > 500 (agentId != 0)' => [
                new DateTime('today + 1 day'), 1, array_merge($saleData, ['bem_bedrag' => 560]),
            ],
            'agent sale and fee < 100 (agentId != 0)' => [
                new DateTime('today + 3 days'), 1, array_merge($saleData, ['bem_bedrag' => 60]),
            ],
        ];
    }

    public function fixtureGetValidSaleData()
    {
        return [
            "lotId" => "8836102",
            "kavelref" => "8836102",
            "reference" => "1",
            "aan_aanhef_kl" => "1",
            "aan_naam_kl" => "A. Abdallah",
            "aan_straat_kl" => "Meidoornstraat",
            "aan_huisnummer_kl" => "115",
            "aan_postcode_kl" => "2665 DA",
            "aan_woonplaats_kl" => "BLEISWYK",
            "aan_land_kl" => "1",
            "aan_tel_kl" => "+31621126655",
            "aan_gsm_kl" => "",
            "aan_provincie_kl" => "12",
            "aan_email_kl" => "Ahmadyousef1313@gmail.com",
            "pakket" => "1",
            "bonus_profile" => "1",
            "bid_id" => "24095062",
            "verkoop_bedrag" => "1900",
            "bem_bedrag" => "300",
            "gehaald" => "0",
            "bedrijvenID" => "2513",
            "make" => "TOYOTA",
            "model" => "VERSO",
        ];
    }

    public function fixtureGetValidConfigClass()
    {
        return new Config([
            Config::EMAIL_SERVICE_API_URL => 'https://www.test.com',
            Config::SALES_AGREEMENT_LOCATION => '//tmp',
            Config::DEALER_SITE_IMPERSONATE_KEY => '123456789',
            Config::DEALER_SITE_URL => 'https://www.dealerdirect.eu',
            Config::VONAGE_API_KEY => 'api-key',
            Config::VONAGE_API_SECRET => 'api-secret',
        ]);
    }
}
