<?php

namespace Dealerdirect\Deliveries\Repositories;

use DateTime;
use Dealerdirect\Deliveries\Exceptions\RepositoryException;

abstract class CarRepository extends AbstractRepository
{
    protected $lotsTable = 'dealer01_ddmain.verzamel_base';

    /**
     * @inheritDoc
     */
    public function setContactDate(int $deliverId, int $countryId, DateTime $dateTime): bool
    {
        $table = 'dealer01_ddmain.verkopen_levering_additional';
        $inserts = [
            'verkopenlevering_id' => $deliverId,
            'country_id' => $countryId,
            'date_contact' => $dateTime,
        ];

        $result = $this->connection->insert($table, $inserts, ['date_contact' => 'datetime']);
        if ($result !== 1) {
            throw RepositoryException::insert($table, $inserts);
        }

        return true;
    }

    /**
     * @inheritDoc
     */
    public function updateLockedSale(int $lotId): bool
    {
        $lockId = $this->connection->fetchOne(
            <<<SQL
            SELECT id FROM dealer01_ddmain.verkopen_geopend
            WHERE kavel = :lotId
            ORDER BY id DESC
            SQL,
            compact('lotId')
        );

        if (empty($lockId)) {
            return true;
        }

        return $this->connection->update(
            'dealer01_ddmain.verkopen_geopend',
            ['verkocht' => 1],
            ['id' => $lockId]
        ) === 1;
    }

    public function updateBids(int $lotId): bool
    {
        return $this->connection->createQueryBuilder()
            ->update('dealer01_ddmain.biedingen_base')
            ->set('status', '"expired"')
            ->where('kavel = :lotId')
            ->andWhere('status = "active"')
            ->setParameters(compact('lotId'))
            ->executeStatement() >= 1
        ;
    }

    /**
     * @inheritDoc
     */
    public function getSaleData(int $bidId): array
    {
        return $this->fetch(
            <<<SQL
            SELECT
                verzamel_base.id AS lotId,
                verzamel_base.id AS kavelref,
                verzamel_base.reference,
                verzamel_base.aanhef aan_aanhef_kl,
                CONCAT_WS(' ', verzamel_base.initialen, verzamel_base.naam) aan_naam_kl,
                verzamel_base.straatnaam aan_straat_kl,
                verzamel_base.huisnummer aan_huisnummer_kl,
                TRIM(CONCAT_WS(' ', verzamel_base.postcodec, verzamel_base.postcodel)) aan_postcode_kl,
                verzamel_base.woonplaats aan_woonplaats_kl,
                verzamel_base.land aan_land_kl,
                verzamel_base.telefoonnummer aan_tel_kl,
                verzamel_base.gsmnummer aan_gsm_kl,
                verzamel_base.provincie aan_provincie_kl,
                verzamel_base.email aan_email_kl,
                ddb.pakket,
                ddb.bonus_profile,
                bb.id bid_id,
                bb.bod verkoop_bedrag,
                bb.comm bem_bedrag,
                bb.gehaald,
                bb.hbieder bedrijvenID,
                me.merkNaam make,
                mo.modelNaam model
            FROM dealer01_ddmain.biedingen_base bb
                INNER JOIN dealer01_ddmain.verzamel_base
                    ON verzamel_base.id = bb.kavel
                INNER JOIN dealer01_ddmain.dealerdirect_bedr ddb
                    ON bb.hbieder = ddb.bedrijvenID
                LEFT JOIN dealer01_webservices.SB_MERKEN me
                    ON me.merkID = verzamel_base.automerk
                LEFT JOIN dealer01_webservices.SB_MODELLEN mo
                    ON mo.modelID = verzamel_base.autotype
                    AND mo.merkID = verzamel_base.automerk
            WHERE bb.id = :bidId
                AND verzamel_base.status != 2
            SQL,
            compact('bidId')
        ) ?: [];
    }
}
