<?php

namespace Dealerdirect\Deliveries\Repositories;

use DateTime;
use Dealerdirect\Deliveries\Exceptions\RepositoryException;
use Dealerdirect\Deliveries\Exceptions\DeliveryException;
use Doctrine\DBAL\Connection;

abstract class AbstractRepository
{
    /** @var string $table - Deliveries table */
    protected $table;

    /** @var string $lotsTable */
    protected $lotsTable;

    /** @var Connection $connection - DBAL connection */
    public $connection;

    /**
     * @param Connection $connection
     */
    public function __construct(Connection $connection)
    {
        $this->connection = $connection;
        if (empty($this->table)) {
            throw new DeliveryException('Error, $table is not defined in Repository!');
        }
    }

    /**
     * Fetches single row and throws exception when data is empty
     *
     * @param string $query
     * @param array $params
     * @return array
     */
    public function fetch(string $query, array $params = [])
    {
        $results = $this->connection->fetchAssociative($query, $params);
        if (empty($results)) {
            throw RepositoryException::emptyData($query, $params);
        }

        return $results;
    }

    public function beginTransaction(): bool
    {
        return $this->connection->beginTransaction();
    }

    public function commit(): bool
    {
        if ($this->connection->commit() === false) {
            throw RepositoryException::commit();
        }

        return true;
    }

    public function rollback(): bool
    {
        return $this->connection->rollback();
    }

    /**
     * Inserts new delivery
     *
     * @param array $data
     * @return integer - the insert ID
     * @throws RepositoryException
     */
    public function create(array $data): int
    {
        if ($this->connection->insert($this->table, $data, ['geboortedatum' => 'datetime']) !== 1) {
            throw RepositoryException::insert($this->table, $data);
        }

        return $this->connection->lastInsertId();
    }

    /**
     * Updates lot to sold; status = 2
     *
     * @param integer $lotId
     * @return boolean
     */
    public function updateStatusToSold(int $lotId): bool
    {
        return $this->connection->update(
            $this->lotsTable,
            ['status' => 2],
            ['id' => $lotId]
        ) === 1;
    }

    /**
     * Updates column agreements_created
     *
     * @param integer $deliverId
     * @param boolean $value
     * @return bool
     */
    public function updateAgreementsCreated(int $deliverId, bool $value): bool
    {
        return $this->connection->update(
            $this->table,
            ['agreements_created' => $value],
            ['id' => $deliverId]
        ) === 1;
    }

    /**
     * Gets data to insert into deliveries table
     *
     * @param integer $bidId
     * @return array
     * @throws RepositoryException
     */
    abstract public function getSaleData(int $bidId): array;

    /**
     * Gets data for sending Agree sale sms.
     *
     * @param integer $deliverId
     * @return array
     * @throws RepositoryException
     */
    abstract public function getSmsData(int $deliverId): array;

    /**
     * Sets contact date for deliveries
     *
     * @param integer $deliverId
     * @param integer $countryId
     * @param DateTime $dateTime
     * @return boolean
     */
    abstract public function setContactDate(int $deliverId, int $countryId, DateTime $dateTime): bool;

    /**
     * Updates locked sale.
     *
     * @param integer $lotId
     * @return boolean
     */
    abstract public function updateLockedSale(int $lotId): bool;

    /**
     * Gets vehicleType for current repository
     *
     * @return string
     */
    abstract public function getVehicleType(): string;

    /**
     * Gets countryCode for current repository
     *
     * @return string
     */
    abstract public function getCountryCode(): string;
}
