<?php

declare(strict_types=1);

namespace Dealerdirect\Deliveries;

use DateTime;
use PHPUnit\Framework\TestCase;
use Dealerdirect\Deliveries\Delivery;
use Dealerdirect\Deliveries\Exceptions\AddSaleException;
use Dealerdirect\Deliveries\Exceptions\DeliveryException;
use Dealerdirect\Deliveries\Repositories\CarNlRepository;
use Dealerdirect\Deliveries\Exceptions\RepositoryException;
use DealerDirect\Generic\Category\Site;
use DealerDirect\Generic\Category\VehicleType;
use Doctrine\DBAL\Connection;

class DeliveryTest extends TestCase
{
    public function testConstructWithVehicleTypeAndCountryCodeShouldThrowExceptionWhenInvalidVehicleTypeOrCountryCode()
    {
        $this->expectException(DeliveryException::class);
        $this->expectExceptionMessage("Invalid branch 'vehicletype_country'.");

        /** @var Connection $connection */
        $connection = $this->createStub(Connection::class);

        Delivery::constructWithVehicleTypeAndCountry('vehicletype', 'country', $connection);
    }

    /**
     * @dataProvider provideValidVehicleTypesAndCountryCodes
     */
    public function testConstructWithVehicleTypeAndCountryShouldReturnNewInstance($vehicleType, $countryCode)
    {
        $expectedInstance = Delivery::class;

        /** @var Connection $connection */
        $connection = $this->createStub(Connection::class);

        $delivery = Delivery::constructWithVehicleTypeAndCountry($vehicleType, $countryCode, $connection);
        $this->assertTrue($delivery instanceof $expectedInstance);
    }

    /**
     * @dataProvider provideValidReferences
     */
    public function testConstructWithReferenceShouldReturnNewInstance(int $reference)
    {
        $expectedInstance = Delivery::class;

        /** @var Connection $connection */
        $connection = $this->createStub(Connection::class);

        $delivery = Delivery::constructWithReference($reference, $connection);
        $this->assertTrue($delivery instanceof $expectedInstance);
    }

    /**
     * @covers Delivery::addSale || RepositoryException
     */
    public function testAddSaleShouldThrowAddSaleExceptionWhenEmptyData()
    {
        $this->expectException(RepositoryException::class);
        $this->expectExceptionMessageMatches('/Empty data with \[Query: ".*?"; Params: bidId: 123456\]/');

        $connection = $this->createStub(Connection::class);
        $connection->method('fetchAssociative')->willReturn(false);
        $connection->method('beginTransaction')->willReturn(true);
        $connection->method('rollback')->willReturn(true);
        /** @var Connection $connection */

        $repository = new CarNlRepository($connection);

        $delivery = new Delivery($repository);
        $delivery->addSale(123456, 1234, '192.168.1.1', new DateTime('14-03-1991'));
    }

    /**
     * @covers Delivery::addSale || DeliveryException
     */
    public function testAddSaleShouldThrowAddSaleExceptionWhenDataMissesRequiredKeys()
    {
        $this->expectException(DeliveryException::class);
        $missingKeys = ['verkoop_bedrag', 'bem_bedrag'];
        foreach ($missingKeys as $missingKey) {
            $this->expectExceptionMessageMatches("/{$missingKey}/");
        }


        $repository = $this->createStub(CarNlRepository::class);
        $saleData = $this->fixtureGetValidSaleData();
        unset($saleData['verkoop_bedrag'], $saleData['bem_bedrag']);
        $repository->method('getSaleData')->willReturn($saleData);

        /** @var CarNlRepository $repository */
        $delivery = new Delivery($repository);
        $delivery->addSale(123456, 1234, '192.168.1.1', new DateTime('14-03-1991'));
    }

    /**
     * @covers Delivery::updateStatusToSold || AddSaleException
     */
    public function testUpdateStatusToSoldShouldRollbackOnErrorAndThrowException()
    {
        $this->expectException(AddSaleException::class);
        $this->expectExceptionMessage('Lot status not updated to sold with [lotId: 8836102].');

        $repository = $this->createStub(CarNlRepository::class);
        $repository->method('getSaleData')->willReturn($this->fixtureGetValidSaleData());
        $repository->method('updateStatusToSold')->willReturn(false);
        $repository->expects($this->once())->method('rollback');

        /** @var CarNlRepository $repository */
        $delivery = new Delivery($repository);
        $delivery->addSale(123456, 1234, '192.168.1.1', new DateTime('14-03-1991'));
    }

    /**
     * @covers Delivery::updateLockedSale || AddSaleException
     */
    public function testUpdateLockedSaleShouldRollbackOnErrorAndThrowException()
    {
        $this->expectException(AddSaleException::class);
        $this->expectExceptionMessage('Could not update locked sale with [lotId: 8836102]');

        $repository = $this->createStub(CarNlRepository::class);
        $repository->method('getSaleData')->willReturn($this->fixtureGetValidSaleData());
        $repository->method('updateStatusToSold')->willReturn(true);
        $repository->method('updateLockedSale')->willReturn(false);
        $repository->expects($this->once())->method('rollback');

        /** @var CarNlRepository $repository */
        $delivery = new Delivery($repository);
        $delivery->addSale(123456, 1234, '192.168.1.1', new DateTime('14-03-1991'));
    }

    /**
     * @covers Delivery::updateLockedSale || AddSaleException
     */
    public function testCreateShouldRollbackOnErrorAndThrowException()
    {
        $this->expectException(RepositoryException::class);
        // $this->expectExceptionMessage('Could not update locked sale with [lotId: 8836102]');

        $repository = $this->createStub(CarNlRepository::class);
        $repository->method('getSaleData')->willReturn($this->fixtureGetValidSaleData());
        $repository->method('updateStatusToSold')->willReturn(true);
        $repository->method('updateLockedSale')->willReturn(true);

        $withData = $this->fixtureGetValidSaleData();
        unset($withData['make'], $withData['model'], $withData['reference'], $withData['lotId']);
        $withData = array_merge($withData, [
            'verkoper' => 1234,
            'ip' => '192.168.1.1',
            'afhandeling_mail' => 1,
            'geboortedatum' => new DateTime('14-03-1991'),
        ]);

        $repository->method('create')->with($withData)->willThrowException(
            RepositoryException::insert('dealer01_ddmain.verkopen_levering', $withData)
        );

        $repository->expects($this->once())->method('rollback');

        /** @var CarNlRepository $repository */
        $delivery = new Delivery($repository);
        $delivery->addSale(123456, 1234, '192.168.1.1', new DateTime('14-03-1991'));
    }

    /**
     * @dataProvider provideContactDateValues
     * @covers Delivery::calculateContactDate
     */
    public function testSetContactDateShouldBeCalledWithCalculatedDatetime($expected, $agentId, $saleData)
    {
        $countryId = 1;
        $repository = $this->createStub(CarNlRepository::class);
        $repository->method('getSaleData')->willReturn($saleData);
        $repository->method('updateStatusToSold')->willReturn(true);
        $repository->method('updateLockedSale')->willReturn(true);
        $repository->method('create')->willReturn(12345);
        $repository->method('getCountryId')->willReturn($countryId);
        $repository->expects($this->once())->method('setContactDate')->with(12345, $countryId, $expected);

        /** @var CarNlRepository $repository */
        $delivery = new Delivery($repository);
        $delivery->addSale(123456, $agentId, '192.168.1.1', new DateTime('14-03-1991'));
    }

    /**
     * @covers Delivery::addSale || AbstractRepository::setContactDate
     */
    public function testAddSaleShouldRollbackAndThrowExceptionWhensetContactDateFailed()
    {
        $this->expectException(RepositoryException::class);

        $repository = $this->createStub(CarNlRepository::class);
        $repository->method('getSaleData')->willReturn($this->fixtureGetValidSaleData());
        $repository->method('updateStatusToSold')->willReturn(true);
        $repository->method('updateLockedSale')->willReturn(true);
        $repository->method('create')->willReturn(12345);

        $table = 'dealer01_ddmain.verkopen_levering_additional';
        $inserts = [
            'verkopenlevering_id' => 12345,
            'country_id' => 1,
            'date_contact' => new DateTime('today'),
        ];
        $repository->method('setContactDate')->willThrowException(RepositoryException::insert($table, $inserts));
        $repository->expects($this->once())->method('rollback');

        /** @var CarNlRepository $repository */
        $delivery = new Delivery($repository);
        $delivery->addSale(123456, 0, '192.168.1.1', new DateTime('14-03-1991'));
    }

    /**
     * @covers Delivery::addSale || AbstractRepository::setContactDate
     */
    public function testAddSaleShouldRollbackAndThrowExceptionWhenCommitFailed()
    {
        $this->expectException(RepositoryException::class);
        $this->expectExceptionMessage('Add sale failed! Could not commit changes');

        $repository = $this->createStub(CarNlRepository::class);
        $repository->method('getSaleData')->willReturn($this->fixtureGetValidSaleData());
        $repository->method('updateStatusToSold')->willReturn(true);
        $repository->method('updateLockedSale')->willReturn(true);
        $repository->method('create')->willReturn(12345);
        $repository->method('setContactDate')->willReturn(true);
        $repository->method('commit')->willThrowException(RepositoryException::commit());

        $repository->expects($this->once())->method('rollback');

        /** @var CarNlRepository $repository */
        $delivery = new Delivery($repository);
        $delivery->addSale(123456, 0, '192.168.1.1', new DateTime('14-03-1991'));
    }

    /**
     * @covers Delivery::addSale
     */
    public function testAddSaleShouldReturnNewDeliverIdOnSuccess()
    {
        $expected = 12345;

        $repository = $this->createStub(CarNlRepository::class);
        $repository->method('getSaleData')->willReturn($this->fixtureGetValidSaleData());
        $repository->method('updateStatusToSold')->willReturn(true);
        $repository->method('updateLockedSale')->willReturn(true);
        $repository->method('create')->willReturn(12345);
        $repository->method('setContactDate')->willReturn(true);
        $repository->method('commit')->willReturn(true);

        /** @var CarNlRepository $repository */
        $delivery = new Delivery($repository);
        $actual = $delivery->addSale(123456, 0, '192.168.1.1', new DateTime('14-03-1991'));

        $this->assertEquals($expected, $actual);
    }

    public function provideValidVehicleTypesAndCountryCodes()
    {
        return [
            'car_nl' => [VehicleType::CAR, 'NL'],
            'car_be' => [VehicleType::CAR, 'BE'],
            'car_de' => [VehicleType::CAR, 'DE'],
            'motorbike_nl' => [VehicleType::MOTOR, 'NL'],
            'motorbike_be' => [VehicleType::MOTOR, 'BE'],
            'motorbike_de' => [VehicleType::MOTOR, 'DE'],
            'scooter_nl' => [VehicleType::SCOOTER, 'NL'],
            'bicycle_nl' => [VehicleType::BICYCLE, 'NL'],
            'marktplaats_nl' => [VehicleType::CAR, 'NL'],
        ];
    }

    public function provideValidReferences()
    {
        return [
            'car_nl' => [Site::REF_CAR_NL],
            'car_be' => [Site::REF_CAR_BE_NL],
            'car_de' => [Site::REF_CAR_DE],
            'motorbike_nl' => [Site::REF_MOTO_NL],
            'motorbike_be' => [Site::REF_MOTO_BE],
            'motorbike_de' => [Site::REF_MOTO_DE],
            'scooter_nl' => [Site::REF_SCOOTER_NL],
            'bicycle_nl' => [Site::REF_BICYCLE_NL],
            'marktplaats_nl' => [Site::REF_CAR_MARKTPLAATS_NL],
        ];
    }

    public function provideContactDateValues()
    {
        $saleData = $this->fixtureGetValidSaleData();

        return [
            'internet sale and high fee (agentId = 0)' => [
                new DateTime('today'), 0, array_merge($saleData, ['bem_bedrag' => 1200]),
            ],
            'internet sale and low fee (agentId = 0)' => [
                new DateTime('today'), 0, array_merge($saleData, ['bem_bedrag' => 50]),
            ],
            'agent sale and fee > 100 (agentId != 0)' => [
                new DateTime('today + 5 day'), 1, array_merge($saleData, ['bem_bedrag' => 120]),
            ],
            'agent sale and fee > 500 (agentId != 0)' => [
                new DateTime('today + 5 day'), 1, array_merge($saleData, ['bem_bedrag' => 560]),
            ],
            'agent sale and fee < 100 (agentId != 0)' => [
                new DateTime('today + 5 day'), 1, array_merge($saleData, ['bem_bedrag' => 60]),
            ],
        ];
    }

    public function fixtureGetValidSaleData()
    {
        return [
            "lotId" => "8836102",
            "kavelref" => "8836102",
            "reference" => "1",
            "aan_aanhef_kl" => "1",
            "aan_naam_kl" => "A. Abdallah",
            "aan_straat_kl" => "Meidoornstraat",
            "aan_huisnummer_kl" => "115",
            "aan_postcode_kl" => "2665 DA",
            "aan_woonplaats_kl" => "BLEISWYK",
            "aan_land_kl" => "1",
            "aan_tel_kl" => "+31621126655",
            "aan_gsm_kl" => "",
            "aan_provincie_kl" => "12",
            "aan_email_kl" => "Ahmadyousef1313@gmail.com",
            "pakket" => "1",
            "bonus_profile" => "1",
            "bid_id" => "24095062",
            "verkoop_bedrag" => "1900",
            "bem_bedrag" => "300",
            "gehaald" => "0",
            "bedrijvenID" => "2513",
            "make" => "TOYOTA",
            "model" => "VERSO",
        ];
    }
}
