<?php

namespace Dealerdirect\Emails\Consumer;

use DateTime;
use Dealerdirect\Emails\Mailer;
use Dealerdirect\Emails\Traits\AanmeldData;
use Dealerdirect\Emails\Traits\ConsumerDomains;
use Dealerdirect\Emails\Traits\EncryptHash;
use DealerDirect\Generic\Category\Country;
use DealerDirect\Generic\Category\EmailType;
use DealerDirect\Generic\Category\Site;
use Dealerdirect\Sdk\Model\Email\SubstitutionKeys as Key;
use Dealerdirect\Sdk\Model\Email\Substitutions;

class ValuationRegistrationSuccessful extends Mailer
{
    use AanmeldData;
    use ConsumerDomains;
    use EncryptHash;

    /** @var string */
    protected string $emailType = EmailType::CONSUMER_VALUATION_FLOW_REGISTRATION_SUCCESSFUL;

    /** @var int */
    protected $countryId;

    /** @var array */
    protected $data;

    /** @var Substitutions */
    protected $substitutions;

    private const REGISTRATION_IWVMAA_NL = 'iwvmaa';
    private const REGISTRATION_IWVMAA_BE = 'iwvmaa_belgie';

    /**
     * @param string $agentName
     * @return void
     */
    public function __invoke(int $countryId = Country::NETHERLANDS)
    {
        $this->countryId = $countryId;
        $this->setData();

        $mileage = number_format($this->data['mileage'], 0, ",", ".");

        $this->substitutions = new Substitutions([
            Key::FIRST_NAME => $this->data['first_name'],
            Key::LAST_NAME => $this->data['last_name'],
            Key::VEHICLE_REGISTRATION_PLATE => strtoupper($this->data['license_plate']),
            Key::VEHICLE_MAKE => $this->data['make'],
            Key::VEHICLE_MODEL => $this->data['model'],
            Key::VEHICLE_MILEAGE => $mileage,
            Key::VALUE_DATE => $this->getValuationDate(),
            Key::URL => $this->getUrl(),
            Key::URL_UNSUBSCRIBE => $this->getUnsubscribeUrl(),
        ]);

        $this->send();
    }

    /**
     * @return string
     */
    public function getUrl(): string
    {
        $reference = (int) $this->mapRegistrationReferenceToSiteReference($this->data['reference']);
        $encryptedId = $this->encryptHash($this->data['lot_id']);

        return sprintf(
            '%s/%s',
            rtrim($this->getDomainByReference($reference), '/'),
            $reference === Site::REF_CAR_MARKTPLAATS_NL
                ? "#/waardebepaling-opvragen?token=$encryptedId"
                : "waardebepaling-opvragen/$encryptedId",
        );
    }

    /**
     * Maps the reference from the registration database to the Site reference.
     */
    private function mapRegistrationReferenceToSiteReference(
        string $registrationReference
    ): int {
        return match ($registrationReference) {
            self::REGISTRATION_IWVMAA_NL => Site::REF_CAR_NL,
            self::REGISTRATION_IWVMAA_BE => Site::REF_CAR_BE_NL,
            default => throw new \Exception("Error, invalid RegistrationReference: '{$registrationReference}'."),
        };
    }

    /**
     * Get the valuation date from the extra data column.
     * If no date is available return todays date.
     */
    private function getValuationDate()
    {
        $extraData = json_decode($this->data['extra_data']);

        return (new DateTime($extraData->valuation_check ?? 'now'))->format('d-m-Y');
    }
}
