<?php

namespace Dealerdirect\Emails\Consumer;

use Dealerdirect\Emails\Mailer;
use DealerDirect\Generic\Category\Site;
use DealerDirect\Generic\Category\Country;
use Dealerdirect\Emails\Traits\EncryptHash;
use Dealerdirect\Emails\Traits\ConsumerData;
use DealerDirect\Generic\Category\EmailType;
use Dealerdirect\Emails\Traits\DeliveryTable;
use DealerDirect\Generic\Category\VehicleType;
use Dealerdirect\Emails\Traits\ConsumerDomains;
use Dealerdirect\Sdk\Model\Email\Substitutions;
use Dealerdirect\Sdk\Model\Email\SubstitutionKeys as Key;

class DealerRatingSurvey extends Mailer
{
    use ConsumerData;
    use ConsumerDomains;
    use EncryptHash;
    use DeliveryTable;

    /** @var string */
    protected $emailType = EmailType::DEALER_RATING_SURVEY;

    protected $query = [
        VehicleType::CAR => [
            'selects' => [
                'vl.id AS deliver_id',
                'bl.voornaam AS agent_name',
                'ddb.bedrijfsnaam AS dealer_name',
            ],
            'table' => ['dealer01_ddmain.verkopen_levering', 'vl'],
            'leftJoin' => [
                ['vl', 'dealer01_ddmain.verzamel_base', 'v', 'vl.kavelref = v.id'],
                ['vl', 'dealer01_ddmain.backoffice_login', 'bl', 'vl.verkoper = bl.user_id'],
                ['vl', 'dealer01_ddmain.dealerdirect_bedr', 'ddb', 'vl.bedrijvenID = ddb.bedrijvenID'],
            ],
            'where' => ['vl.id = :id'],
        ],
        VehicleType::MOTOR => [
            'selects' => [
                'vl.id AS deliver_id',
                'bl.voornaam AS agent_name',
                'ddb.bedrijfsnaam AS dealer_name',
            ],
            'table' => ['dealer01_ddmoto.BO_Verkopen_NL', 'vl'],
            'leftJoin' => [
                ['vl', 'dealer01_ddmoto.BO_Motorfiets', 'v', 'vl.kavelref = v.id'],
                ['vl', 'dealer01_ddmain.backoffice_login', 'bl', 'vl.verkoper = bl.user_id'],
                ['vl', 'dealer01_ddmain.dealerdirect_bedr', 'ddb', 'vl.bedrijvenID = ddb.bedrijvenID'],
            ],
            'where' => ['vl.id = :id'],
        ],
    ];

    /**
     * @param string $agentName
     * @return void
     */
    public function __invoke(int $countryId = Country::NETHERLANDS)
    {
        $this->countryId = $countryId;
        $this->setTable();
        $this->setData();

        $this->substitutions = new Substitutions([
            Key::FIRST_NAME => $this->data['first_name'],
            Key::LAST_NAME => $this->data['last_name'],
            Key::DEALER_NAME => $this->data['dealer_name'],
            Key::AGENT_NAME => $this->getAgentName(),
            Key::AGENT_PHONE => $this->getGroupPhoneByLotId(),
            Key::URL => $this->getUrl(),
            Key::URL_UNSUBSCRIBE => $this->getUnsubscribeUrl(),
        ]);

        $this->send();
    }

    /**
     * @return string
     */
    private function getUrl(): string
    {
        $domain = $this->getDomainByReference((int) $this->data['reference']);
        $encryptedId = $this->encryptHash($this->data['deliver_id']);
        $endpoint = "/afhandeling-levering/survey/{$encryptedId}";
        if (in_array((int) $this->data['reference'], [Site::REF_CAR_DE, Site::REF_MOTO_DE], true)) {
            $endpoint = "/lieferung/survey/{$encryptedId}";
        } elseif (((int) $this->data['reference']) === Site::REF_CAR_MARKTPLAATS_NL) {
            $endpoint = "/#/dealer-beoordeling?token=$encryptedId";
        }

        return sprintf('%s%s', rtrim($domain, '/'), $endpoint);
    }
}
