<?php

namespace Dealerdirect\Emails\Consumer;

use Dealerdirect\Emails\Mailer;
use DealerDirect\Generic\Category\Site;
use DealerDirect\Generic\Category\Country;
use Dealerdirect\Emails\Traits\EncryptHash;
use Dealerdirect\Emails\Traits\ConsumerData;
use DealerDirect\Generic\Category\EmailType;
use Dealerdirect\Emails\Traits\DeliveryTable;
use DealerDirect\Generic\Category\VehicleType;
use Dealerdirect\Emails\Traits\ConsumerDomains;
use Dealerdirect\Sdk\Model\Email\Substitutions;
use Dealerdirect\Sdk\Model\Email\SubstitutionKeys as Key;

class DeliveryHandling extends Mailer
{
    use ConsumerData;
    use ConsumerDomains;
    use EncryptHash;
    use DeliveryTable;

    public const MANUALLY = 0;
    public const FIRST_FOLLOW_UP = 8;
    public const SECOND_FOLLOW_UP = 72;

    /** @var string */
    protected $emailType;

    protected $query = [
        VehicleType::CAR => [
            'selects' => [
                'vl.id as deliver_id',
                'vl.kavelref as lot_id',
                'ddb.bedrijfsnaam AS dealer_name',
                'ddb.plaats AS dealer_city',
                'vl.verkoper AS agent',
            ],
            'table' => ['dealer01_ddmain.verkopen_levering', 'vl'],
            'leftJoin' => [
                ['vl', 'dealer01_ddmain.verzamel_base', 'v', 'vl.kavelref = v.id'],
                ['vl', 'dealer01_ddmain.dealerdirect_bedr', 'ddb', 'vl.bedrijvenID = ddb.bedrijvenID'],
            ],
            'where' => ['vl.id = :id'],
        ],
        VehicleType::MOTOR => [
            'selects' => [
                'vl.id as deliver_id',
                'vl.kavelref as lot_id',
                'ddb.bedrijfsnaam AS dealer_name',
                'ddb.plaats AS dealer_city',
                'vl.verkoper AS agent',
            ],
            'table' => ['dealer01_ddmoto.BO_Verkopen_NL', 'vl'],
            'leftJoin' => [
                ['vl', 'dealer01_ddmoto.BO_Motorfiets', 'v', 'vl.kavelref = v.id'],
                ['vl', 'dealer01_ddmain.dealerdirect_bedr', 'ddb', 'vl.bedrijvenID = ddb.bedrijvenID'],
            ],
            'where' => ['vl.id = :id'],
        ],
    ];

    /**
     * @param integer $countryId
     * @param integer $followUp
     * @return void
     */
    public function __invoke(int $countryId = Country::NETHERLANDS, int $followUp = self::MANUALLY)
    {
        $this->countryId = $countryId;
        $this->setTable();
        $this->setData();
        $this->setEmailType($followUp);

        $this->substitutions = new Substitutions([
            Key::FIRST_NAME => $this->data['first_name'],
            Key::LAST_NAME => $this->data['last_name'],
            Key::LOT_NUMBER => $this->data['lot_id'],
            Key::VEHICLE_MAKE => $this->data['make'],
            Key::VEHICLE_MODEL => $this->data['model'],
            Key::URL => $this->getUrl(),
            Key::CUSTOMER_SERVICE_PHONE => $this->getCustomerServicePhone(),
            Key::DEALER_NAME => $this->data['dealer_name'],
            Key::DEALER_ADDRESS => $this->data['dealer_city'],
            Key::AGENT_PHONE => $this->getGroupPhoneByLotId(),
            Key::AGENT_WHATSAPP => $this->getGroupWhatsappByLotId(),
            Key::URL_UNSUBSCRIBE => $this->getUnsubscribeUrl(),
        ]);

        $this->send();
    }

    /**
     * @return string
     */
    private function getUrl(): string
    {
        $domain = $this->getDomainByReference((int) $this->data['reference']);
        $encryptedId = $this->encryptHash($this->data['deliver_id']);
        $reference = (int) $this->data['reference'];

        $endpoint = "/afhandeling-levering/{$encryptedId}";
        if (in_array($reference, [Site::REF_CAR_DE, Site::REF_MOTO_DE], true)) {
            $endpoint = "/lieferung/{$encryptedId}";
        } elseif ($reference === Site::REF_CAR_MARKTPLAATS_NL) {
            $endpoint = "/#/afhandeling-levering?token={$encryptedId}";
        }

        return sprintf('%s%s', rtrim($domain, '/'), $endpoint);
    }

    /**
     * @param integer $followUp
     * @return string
     */
    private function setEmailType(int $followUp): void
    {
        $types = [
            self::FIRST_FOLLOW_UP => EmailType::DELIVERY_HANDLING_FIRST,
            self::SECOND_FOLLOW_UP => EmailType::DELIVERY_HANDLING_SECOND,
        ];

        $this->emailType = $types[$followUp] ?? EmailType::DELIVERY_HANDLING_MANUAL;
        if (in_array($this->vehicleType, [VehicleType::SCOOTER, VehicleType::MOTOR, VehicleType::BICYCLE])) {
            $this->emailType = $this->emailType === EmailType::DELIVERY_HANDLING_MANUAL
                ? EmailType::DELIVERY_HANDLING_FIRST
                : $this->emailType;
        }
    }
}
