<?php

namespace Dealerdirect\Emails\Consumer;

use Dealerdirect\Emails\Mailer;
use DealerDirect\Generic\Category\Country;
use Dealerdirect\Emails\Traits\ConsumerDeliveryData;
use Dealerdirect\Emails\Traits\ConsumerDomains;
use Dealerdirect\Emails\Traits\EncryptHash;
use DealerDirect\Generic\Category\EmailType;
use DealerDirect\Generic\Category\Site;
use Dealerdirect\Sdk\Model\Email\Substitutions;
use Dealerdirect\Sdk\Model\Email\SubstitutionKeys as Key;

class DeliveryHandling extends Mailer
{
    use ConsumerDeliveryData;
    use ConsumerDomains;
    use EncryptHash;

    public const MANUALLY = 0;
    public const FIRST_FOLLOW_UP = 8;
    public const SECOND_FOLLOW_UP = 72;

    /** @var string */
    protected $emailType;

    /**
     * @param integer $countryId
     * @param integer $followUp
     * @return void
     */
    public function __invoke(int $countryId = Country::NETHERLANDS, int $followUp = self::MANUALLY)
    {
        $this->countryId = $countryId;
        $this->setData();
        $this->setEmailType($followUp);
        $this->addRecipient("{$this->data['first_name']} {$this->data['last_name']}", $this->data['email']);

        $this->substitutions = new Substitutions([
            Key::FIRST_NAME => $this->data['first_name'],
            Key::LAST_NAME => $this->data['last_name'],
            Key::LOT_NUMBER => $this->data['lot_id'],
            Key::VEHICLE_MAKE => $this->data['make'],
            Key::VEHICLE_MODEL => $this->data['model'],
            Key::URL => $this->getUrl(),
            Key::CUSTOMER_SERVICE_PHONE => $this->getCustomerServicePhone(),
            Key::DEALER_NAME => $this->data['dealer_name'],
            Key::DEALER_ADDRESS => $this->data['dealer_city'],
        ]);

        $this->send();
    }

    /**
     * @return string
     */
    private function getUrl(): string
    {
        $domain = $this->getDomainByReference((int) $this->data['reference']);
        $encryptedId = $this->encryptHash($this->data['deliver_id']);
        $endpoint = "/afhandeling-levering/{$encryptedId}";
        if (in_array((int) $this->data['reference'], [Site::REF_CAR_DE, Site::REF_MOTO_DE], true)) {
            $endpoint = "/lieferung/{$encryptedId}";
        }

        return sprintf('https://%s%s', rtrim($domain, '/'), $endpoint);
    }

    /**
     * @param integer $followUp
     * @return string
     */
    private function setEmailType(int $followUp): void
    {
        $types = [
            self::FIRST_FOLLOW_UP => EmailType::DELIVERY_HANDLING_FIRST,
            self::SECOND_FOLLOW_UP => EmailType::DELIVERY_HANDLING_SECOND,
        ];

        $this->emailType = $types[$followUp] ?? EmailType::DELIVERY_HANDLING_MANUAL;
    }
}
