<?php

namespace Dealerdirect\Emails\Consumer;

use Dealerdirect\Emails\Exceptions\EmptyInternalPriceException;
use Dealerdirect\Emails\Mailer;
use DealerDirect\Generic\Category\Locale;
use Dealerdirect\Emails\Traits\ConsumerData;
use DealerDirect\Generic\Category\EmailType;
use DealerDirect\Generic\Category\VehicleType;
use Dealerdirect\Emails\Traits\ConsumerDomains;
use Dealerdirect\Emails\Traits\EncryptHash;
use DealerDirect\Generic\Category\LeadNurtureStatus;
use DealerDirect\Generic\Category\VehicleTypeCarBodyType;
use Dealerdirect\Sdk\Model\Email\SubstitutionKeys as Key;
use Dealerdirect\Sdk\Model\Email\Substitutions;

class Nurture extends Mailer
{
    use ConsumerData;
    use ConsumerDomains;
    use EncryptHash;

    protected $emailType;

    protected $query = [
        VehicleType::CAR => [
            'selects' => [
                'v.carrosserie AS bodywork',
                'v.lead_nurture_status',
                'internal_price.value AS internal_price',
                'v.prijsverwacht as asking_price',
                'v.expdatum AS expire_date',
            ],
            'leftJoin' => [
                ['v', 'internalPriceJoin', 'internal_price', 'internal_price.vehicle_id = v.id'],
            ],
            'where' => ['v.status = 1'],
        ],
        VehicleType::MOTOR => [
            'selects' => [
                'v.lead_nurture_status',
                'v.verwachteprijsIntern AS internal_price',
                'v.verwachteprijs as asking_price',
                'exp.expdatum AS expire_date',
            ],
            'leftJoin' => [
                ['v', 'dealer01_ddmoto.BO_Motorfiets_Exp', 'exp', 'exp.voertuigID = v.id'],
            ],
            'where' => ['v.status = 1'],
        ],
    ];

    public function __invoke()
    {
        $this->setData();
        $this->setEmailType();

        $substitutions = [
            Key::FIRST_NAME => $this->data['first_name'],
            Key::LAST_NAME => $this->data['last_name'],
            Key::LOT_NUMBER => $this->data['lot_id'],
            Key::VEHICLE_MAKE => $this->data['make'],
            Key::VEHICLE_MODEL => $this->data['model'],
            Key::AGENT_PHONE => $this->getGroupPhoneByLotId(),
            Key::VALUE_URL => $this->getValuationInvitationUrl(),
        ];

        switch ($this->emailType) {
            case EmailType::LEAD_NURTURE_VALUATION:
                if (empty($this->data['internal_price'])) {
                    $this->emailType = EmailType::LEAD_NURTURE_CURIOUS;
                    break;
                }

                $substitutions += $this->calculateNurtureValuations($this->data['internal_price']);
                break;
            case EmailType::LEAD_NURTURE_ASKING_PRICE:
                if (empty($this->data['asking_price'])) {
                    $this->emailType = EmailType::LEAD_NURTURE_ASKING_PRICE_SECOND;
                }
                break;
        }

        $this->substitutions = new Substitutions($substitutions);
        $this->overwriteVehicleType();
        $this->send();
    }

    private function overwriteVehicleType()
    {
        if ($this->vehicleType === VehicleType::CAR) {
            switch ((int) $this->data['bodywork']) {
                case VehicleTypeCarBodyType::CAMPER:
                    $this->vehicleType = VehicleType::CAMPER;
                    break;
                case VehicleTypeCarBodyType::CARAVAN:
                    $this->vehicleType = VehicleType::CARAVAN;
                    break;
            }
        }
    }

    private function calculateNurtureValuations($internalPrice)
    {
        $valuation = [Key::VALUE_AVG => $internalPrice];
        if ($internalPrice <= 1000) {
            $valuation[Key::VALUE_MIN] = round($internalPrice * 0.6);
            $valuation[Key::VALUE_MAX] = round($internalPrice * 1.25);
        } elseif ($internalPrice <= 5000) {
            $valuation[Key::VALUE_MIN] = round($internalPrice * 0.7);
            $valuation[Key::VALUE_MAX] = round($internalPrice * 1.1);
        } elseif ($internalPrice <= 10000) {
            $valuation[Key::VALUE_MIN] = round($internalPrice * 0.8);
            $valuation[Key::VALUE_MAX] = round($internalPrice * 1.1);
        } else {
            $valuation[Key::VALUE_MIN] = round($internalPrice * 0.9);
            $valuation[Key::VALUE_MAX] = round($internalPrice * 1.1);
        }

        return $valuation;
    }

    private function setEmailType()
    {
        $types = [
            LeadNurtureStatus::NURTURE_CURIOUS => EmailType::LEAD_NURTURE_CURIOUS,
            LeadNurtureStatus::NURTURE_VALUATION => EmailType::LEAD_NURTURE_VALUATION,
            LeadNurtureStatus::NURTURE_MORE_TIME => EmailType::LEAD_NURTURE_MORE_TIME,
            LeadNurtureStatus::NURTURE_ASKING_PRICE => EmailType::LEAD_NURTURE_ASKING_PRICE,
        ];

        if (!array_key_exists($this->data['lead_nurture_status'], $types)) {
            throw new \Exception("Error, invalid nurture status '{$this->data['lead_nurture_status']}', {$this->id}.");
        }

        $this->emailType = $types[$this->data['lead_nurture_status']];
    }

    private function getValuationInvitationUrl(): string
    {
        return sprintf(
            '%s/%s/%s',
            $this->getDomainByReference((int) $this->data['reference']),
            $this->locale === Locale::DE_DE ? 'wertschatzung' : 'waardebepaling',
            $this->encryptHash($this->data['lot_id'])
        );
    }
}
