<?php

namespace Dealerdirect\Emails;

use ForceUTF8\Encoding;
use Doctrine\DBAL\Connection;
use Dealerdirect\Emails\Helper;
use Doctrine\DBAL\Query\QueryBuilder;
use DealerDirect\Generic\Category\Site;
use Dealerdirect\Sdk\Model\Email\Email;
use DealerDirect\Generic\Category\Locale;
use Dealerdirect\Sdk\Model\Email\Recipient;
use DealerDirect\Sdk\Facade\Email as Client;
use DealerDirect\Generic\Category\VehicleType;
use Dealerdirect\Sdk\Model\Email\Substitutions;
use DealerDirect\Generic\Util\PhoneNumberFormatting;
use DealerDirect\Generic\Repository\CustomerServiceInformation;
use Dealerdirect\Sdk\Model\Email\SubstitutionKeys as Key;

abstract class Mailer
{
    /** @var Connection */
    protected $dbal;

    /** @var string */
    protected $vehicleType;

    /** @var string */
    protected $locale;

    /** @var integer */
    protected $id;

    /** @var array */
    private $recipients = [
        'to' => [],
        'cc' => [],
        'bcc' => [],
    ];

    /** @var Substitutions */
    protected $substitutions;

    /** @var Client */
    private $client;

    /** @var array */
    protected $data;

    /** @var Helper */
    protected $helper;

    /**
     * Mailer Constructor
     *
     * @param Client $client
     * @param Connection $dbal
     * @param string $vehicleType
     * @param integer $id
     */
    public function __construct(Client $client, Connection $dbal, string $vehicleType, int $id = null)
    {
        $this->client = $client;
        $this->dbal = $dbal;
        $this->vehicleType = $vehicleType;
        $this->helper = new Helper($dbal, $vehicleType);
        $this->isMotor = in_array($vehicleType, [VehicleType::MOTOR, VehicleType::SCOOTER]);
        $this->id = $id;
    }

    public function getEmailType()
    {
        return $this->emailType;
    }

    public function getData()
    {
        return $this->data;
    }

    /**
     * @param integer $reference
     * @return void
     */
    public function send(): void
    {
        if ($this->locale === Locale::DE_DE && isset($this->data['salutation'])) {
            $substitutions = $this->substitutions->getSubstitutions();
            $substitutions[Key::SALUTATION] = (int) $this->data['salutation'] === 2
                ? 'Sehr geehrte Frau'
                : 'Sehr geehrter Herr';
            $this->substitutions = new Substitutions($substitutions);
        }

        if (empty($this->recipients['to'])) {
            $this->setDefaultRecipient();
        }

        try {
            $email = new Email($this->locale, $this->vehicleType, $this->emailType, $this->substitutions);
            $recipients = $this->recipients;
            $this->client->sendEmail($email, $recipients['to'], $recipients['cc'], $recipients['bcc']);
        } catch (\Exception $exception) {
            throw new \Exception("Error in sending ConsumerEmail for lotId '{$this->id}'.", 0, $exception);
        }
    }

    private function setDefaultRecipient()
    {
        $data = $this->data;
        if (!empty($data['first_name']) && !empty($data['first_name']) && !empty($data['email'])) {
            $this->addRecipient("{$this->data['first_name']} {$this->data['last_name']}", $this->data['email']);
        }
    }

    /**
     * Sets data required by email
     *
     * @return void
     */
    protected function setData(array $params = null)
    {
        $query = $this->buildQuery();
        $fetch = $this->dbal->fetchAssociative($query, $params ?? [':id' => $this->id]);
        if (empty($fetch)) {
            $class = get_called_class();
            throw new \Exception("Error, could not fetch email-data for '$class' with id '{$this->id}'.");
        }

        $this->data = Encoding::fixUTF8($fetch);
        $this->locale = $this->getLocale();
        if ((int) ($this->data['reference'] ?? false) === Site::REF_SCOOTER_NL) {
            $this->vehicleType = VehicleType::SCOOTER;
        }
    }

    private function buildQuery(): QueryBuilder
    {
        // Get base vehicleType to get query. As scooter or recreation does not have a specific query.
        $vehicleType = $this->isMotor ? VehicleType::MOTOR : VehicleType::CAR;
        $queryOptions = ($this->query[$vehicleType] ?? $this->defaultQuery[$vehicleType]);
        if (isset($this->query) && isset($this->defaultQuery)) {
            $mainQuery = $this->query[$vehicleType] ?? [];
            $defaultQuery = $this->defaultQuery[$vehicleType] ?? [];
            $queryOptions = array_merge_recursive($mainQuery, $defaultQuery);
            $queryOptions['table'] = $mainQuery['table'] ?? $defaultQuery['table'];
        }
        if (empty($queryOptions)) {
            throw new \Exception('Error, no query defined!');
        }

        $query = $this->dbal->createQueryBuilder()
            ->from(...$queryOptions['table'])
            ->select(...$queryOptions['selects'])
            ->where(...$queryOptions['where']);

        foreach (['leftJoin', 'innerJoin'] as $type) {
            $query = $this->loopOverJoins($query, $queryOptions, $type);
        }

        return $query;
    }

    private function loopOverJoins(QueryBuilder $query, array $queryOptions, string $type): QueryBuilder
    {
        foreach ($queryOptions[$type] ?? [] as $join) {
            if (is_string($join[1]) && method_exists(Helper::class, $join[1])) {
                $joinQuery = $this->helper->{$join[1]}($this->vehicleType);
                $join[1] = "($joinQuery)";
            }

            $query->$type(...$join);
        }

        return $query;
    }

    /**
     * @param string $name
     * @param string $email
     * @param string $recipientType
     * @return void
     */
    protected function addRecipient(string $name, string $email, string $recipientType = 'to')
    {
        if (!in_array($recipientType, ['to', 'cc', 'bcc'])) {
            throw new \Exception("Error, invalid recipientType '$recipientType'. Valid types: ['to', 'cc', 'bcc'].");
        } elseif (filter_var($email, FILTER_VALIDATE_EMAIL) === false) {
            throw new \Exception("Error, invalid email address '{$email}'.");
        }

        // Check if it already exists
        $exists = false;
        foreach ($this->recipients[$recipientType] as $recipient) {
            if ($recipient->getEmailAddress() === $email) {
                $exists = true;
            }
        }

        if (!$exists) {
            $this->recipients[$recipientType][] = new Recipient(trim($email), trim($name));
        }
    }

    /**
     * @param string $phoneNumber
     * @return string
     */
    protected function formatPhone(string $phoneNumber): string
    {
        $phoneFormat = new PhoneNumberFormatting((int) $this->data['country_id']);

        return $phoneFormat->formatPhoneNumberToNational($phoneNumber);
    }

    /**
     * @return string
     */
    protected function getCustomerServicePhone(): string
    {
        $customerService = new CustomerServiceInformation($this->vehicleType);

        return $this->formatPhone(
            $customerService->getCustomerServicePhoneNumber((int) $this->data['reference'])
        );
    }

    /**
     * @return string
     */
    private function getLocale(): string
    {
        $locales = [
            Locale::NL_NL => [
                Site::REF_CAR_NL, Site::REF_MOTO_NL, SITE::REF_SCOOTER_NL
            ],
            Locale::NL_BE => [
                Site::REF_CAR_BE_NL, Site::REF_MOTO_BE,
            ],
            Locale::DE_DE => [
                Site::REF_CAR_DE, Site::REF_MOTO_DE,
            ],
        ];

        foreach ($locales as $locale => $references) {
            if (in_array((int) $this->data['reference'], $references, true)) {
                return $locale;
            }
        }

        return Locale::NL_NL;
    }
}
