<?php

namespace Dealerdirect\Emails\Consumer;

use Dealerdirect\Emails\Mailer;
use Dealerdirect\Emails\Traits\ConsumerData;
use Dealerdirect\Emails\Traits\ConsumerDomains;
use Dealerdirect\Emails\Traits\EncryptHash;
use DealerDirect\Generic\Category\EmailType;
use DealerDirect\Generic\Category\Locale;
use DealerDirect\Generic\Category\Site;
use DealerDirect\Generic\Category\VehicleType;
use Dealerdirect\Sdk\Model\Email\SubstitutionKeys as Key;
use Dealerdirect\Sdk\Model\Email\Substitutions;

class HighestBid extends Mailer
{
    use ConsumerData;
    use ConsumerDomains;
    use EncryptHash;

    protected $emailType;

    protected $query = [
        VehicleType::CAR => [
            'selects' => [
                'highest_bid.id AS bid_id',
                'highest_bid.hbieder AS company_id',
                'highest_bid.bod AS bid',
                'highest_bid.comm AS commission',
                'internal_price.value AS internal_price',
                'v.expdatum AS expire_date',
            ],
            'leftJoin' => [
                ['v', 'highestBidJoin', 'highest_bid', 'highest_bid.kavel = v.id'],
                ['v', 'internalPriceJoin', 'internal_price', 'internal_price.vehicle_id = v.id'],
            ],
            'where' => [
                'v.status = 1',
                'highest_bid.status = "active" OR highest_bid.id IS NULL'
            ],
        ],
        VehicleType::MOTOR => [
            'selects' => [
                'highest_bid.id AS bid_id',
                'highest_bid.bedrijvenID AS company_id',
                'highest_bid.bedrag AS bid',
                'highest_bid.com AS commission',
                'v.verwachteprijsIntern AS internal_price',
                'exp.expdatum AS expire_date',
                'CG.callgroup_id'
            ],
            'leftJoin' => [
                ['v', 'highestBidJoin', 'highest_bid', 'highest_bid.voertuigID = v.id'],
                ['v', 'dealer01_ddmoto.BO_Motorfiets_Exp', 'exp', 'exp.voertuigID = v.id'],
                ['v', 'dealer01_ddmoto.call_groups_moto', 'CG', 'CG.kavelref = v.id'],
            ],
            'where' => [
                'v.status = 1',
                'highest_bid.status = "active" OR highest_bid.voertuigID IS NULL'
            ],
        ],
    ];

    public function __invoke()
    {
        $this->setData();
        $this->emailType = $this->isOffendingBid() ? EmailType::HIGHEST_BID_OFFENDING : EmailType::HIGHEST_BID;

        $this->substitutions = new Substitutions([
            Key::FIRST_NAME => $this->data['first_name'],
            Key::LAST_NAME => $this->data['last_name'],
            Key::LOT_NUMBER => $this->data['lot_id'],
            Key::VEHICLE_MAKE => $this->data['make'],
            Key::VEHICLE_MODEL => $this->data['model'],
            Key::AGENT_NAME => $this->getAgentName(),
            Key::AGENT_PHONE => $this->getGroupPhoneByLotId(),
            Key::URL => $this->getUrl(),
            Key::URL_UNSUBSCRIBE => $this->getUnsubscribeUrl(),
        ]);

        $this->send();
    }

    private function isOffendingBid()
    {
        $bid = (int) $this->data['bid'] - (int) $this->data['commission'];
        if ($this->data['internal_price'] === 0 || $this->data['internal_price'] === null) {
            return true;
        }

        $result = false;

        if ($this->data['internal_price'] > 6000) {
            $result = $bid <= ($this->data['internal_price'] * 0.75);
        } elseif ($this->data['internal_price'] >= 1000) {
            $result = $bid <= ($this->data['internal_price'] * 0.65);
        } elseif ($this->data['internal_price'] < 1000) {
            $result = $bid <= ($this->data['internal_price'] * 0.25);
        }

        if ($bid > ($this->data['internal_price'] * 1.5)) {
            $result = false;
        }

        return $result;
    }

    private function getUrl(): string
    {
        $reference = (int) $this->data['reference'];
        $domain = $this->getDomainByReference($reference);
        if ($reference === Site::REF_CAR_MARKTPLAATS_NL) {
            return sprintf('%s/#/aanbod?token=%s', $domain, $this->encryptHash($this->data['lot_id']));
        }

        return sprintf(
            '%s/%s/%s/%s',
            $domain,
            $this->locale === Locale::DE_DE ? 'gebote' : 'biedingen',
            $this->data['lot_id'],
            // TODO: Replace sha1 with EncryptHash
            sha1($this->data['email'] . $this->data['expire_date'])
        );
    }
}
