<?php

namespace Dealerdirect\Emails\Traits;

use DealerDirect\Generic\Category\Site;

trait ConsumerDomains
{
    /** @var array */
    private $requiredDomains = [
        'carNl', 'carBe', 'carDe', 'motorNl', 'motorBe', 'motorDe', 'scooterNl', 'bicycleNl', 'marktplaatsNl',
    ];

    /** @var array */
    protected $domains;

    /**
     * Sets consumer domains
     *
     * @param array $domains
     * @return void
     */
    public function setDomains(array $domains)
    {
        foreach ($this->requiredDomains as $domainKey) {
            if (empty($domains[$domainKey])) {
                throw new \Exception("Error, missing domain '$domainKey'.");
            }

            $this->domains[$domainKey] = $domains[$domainKey];
        }
    }

    protected function getNameByReference(int $reference)
    {
        $references = [
            Site::REF_CAR_NL => 'ikwilvanmijnautoaf.nl',
            Site::REF_CAR_BE_NL => 'ikwilvanmijnautoaf.be',
            Site::REF_CAR_DE => 'ichwillmeinautoloswerden.de',
            Site::REF_MOTO_NL => 'ikwilvanmijnmotoraf.nl',
            Site::REF_MOTO_BE => 'ikwilvanmijnmotoraf.be',
            Site::REF_MOTO_DE => 'ichwillmeinmotorradloswerden.de',
            Site::REF_SCOOTER_NL => 'ikwilvanmijnscooteraf.nl',
            Site::REF_BICYCLE_NL => 'ikwilvanmijnfietsaf.nl',
            Site::REF_CAR_MARKTPLAATS_NL => 'Marktplaats Auto Verkoopservice'
        ];

        $this->checkReferences($references, $reference);

        return $references[$reference];
    }

    /**
     * Gets domain by reference
     *
     * @param integer $reference
     * @return string
     */
    protected function getDomainByReference(int $reference)
    {
        $references = [
            Site::REF_CAR_NL => 'carNl',
            Site::REF_CAR_BE_NL => 'carBe',
            Site::REF_CAR_DE => 'carDe',
            Site::REF_MOTO_NL => 'motorNl',
            Site::REF_MOTO_BE => 'motorBe',
            Site::REF_MOTO_DE => 'motorDe',
            Site::REF_SCOOTER_NL => 'scooterNl',
            Site::REF_BICYCLE_NL => 'bicycleNl',
            Site::REF_CAR_MARKTPLAATS_NL => 'marktplaatsNl',
        ];

        $this->checkReferences($references, $reference);
        $parsedUrl = parse_url($this->domains[$references[$reference]]);

        return rtrim(
            sprintf(
                '%s://%s%s%s%s',
                $parsedUrl['scheme'] ?? 'https',
                $parsedUrl['host'] ?? '',
                isset($parsedUrl['port']) ? ":{$parsedUrl['port']}" : '',
                $parsedUrl['path'] ?? '',
                isset($parsedUrl['query']) ? "?{$parsedUrl['query']}" : '',
            ),
            '/'
        );
    }

    /**
     * @param array $references
     * @param integer $reference
     * @return void
     * @throws Exception
     */
    private function checkReferences(array $references, int $reference)
    {
        if (!array_key_exists($reference, $references)) {
            $available = implode(', ', array_keys($references));

            throw new \Exception("Error, invalid reference '$reference'. Available: '$available'");
        }
    }

    protected function getUnsubscribeUrl(): string
    {
        $reference = (int) $this->data['reference'];
        if ($reference !== Site::REF_CAR_MARKTPLAATS_NL) {
            return '';
        }

        $domain = $this->getDomainByReference($reference);
        $encryptedId = $this->encryptHash($this->id);
        $endPoint = "/#/uitschrijven?token={$encryptedId}";

        return sprintf('%s%s', rtrim($domain, '/'), $endPoint);
    }
}
