<?php

namespace Dealerdirect\Emails\Traits;

use Doctrine\DBAL\Query\QueryBuilder;
use DealerDirect\Generic\Category\VehicleType;

/**
 * @property Connection $dbal
 */
trait ConsumerBidData
{
    use ConsumerData {
        ConsumerData::carQuery as parentCarQuery;
        ConsumerData::motorQuery as parentMotorQuery;
    }

    /**
     * Creates subQuery join for highest bid
     *
     * @param string $vehicleType
     * @return QueryBuilder
     */
    protected function highestBidJoin(string $vehicleType = VehicleType::CAR): QueryBuilder
    {
        list($table, $lotColumn, $bidColumn) = $vehicleType === VehicleType::MOTOR
            ? ['dealer01_ddmoto.BO_Biedhistorie', 'voertuigID', 'bedrag']
            : ['dealer01_ddmain.biedingen_base', 'kavel', 'bod'];

        return $this->dbal->createQueryBuilder()
            ->select('t1.*')
            ->from($table, 't1')
            ->leftJoin('t1', $table, 't2', <<<SQL
                t1.$lotColumn = t2.$lotColumn
                AND t1.status = t2.status
                AND t1.$bidColumn < t2.$bidColumn
            SQL)
            ->where("t2.$lotColumn IS NULL")
            ->where('t1.status = "active"');
    }

    /**
     * @return QueryBuilder
     */
    protected function carQuery(): QueryBuilder
    {
        $queryBuilder = $this->parentCarQuery();
        $highestBidJoin = $this->highestBidJoin(VehicleType::CAR);
        $table = $queryBuilder->getQueryPart('from')[0];
        $table = $table['alias'] ?? $table['table'];

        return $queryBuilder
            ->addSelect([
                'highest_bid.comm as commission',
                'highest_bid.bod as highest_bid',
            ])
            ->leftJoin($table, "($highestBidJoin)", 'highest_bid', "highest_bid.kavel = $table.id");
    }

    /**
     * @return QueryBuilder
     */
    protected function motorQuery(): QueryBuilder
    {
        $queryBuilder = $this->parentMotorQuery();
        $highestBidJoin = $this->highestBidJoin(VehicleType::MOTOR);
        $table = $queryBuilder->getQueryPart('from')[0];
        $table = $table['alias'] ?? $table['table'];

        return $queryBuilder
            ->addSelect([
                'highest_bid.com as commission',
                'highest_bid.bedrag as highest_bid',
            ])
            ->leftJoin($table, "($highestBidJoin)", 'highest_bid', "highest_bid.voertuigID = $table.id");
    }
}
