<?php

namespace Dealerdirect\Emails\Dealer;

use Dealerdirect\Emails\Mailer;
use DealerDirect\Generic\Category\Locale;
use DealerDirect\Generic\Category\Country;
use Dealerdirect\Emails\Traits\DealerDomain;
use Dealerdirect\Emails\Traits\DeliveryTable;
use Dealerdirect\Emails\Traits\EncryptHash;
use DealerDirect\Generic\Category\EmailType;
use DealerDirect\Generic\Category\VehicleType;
use Dealerdirect\Sdk\Model\Email\Substitutions;
use Dealerdirect\Sdk\Model\Email\SubstitutionKeys as Key;

class DeliveryStatusRequest extends Mailer
{
    use DealerDomain;
    use DeliveryTable;
    use EncryptHash;

    private const ALLOWED_LOCALES = [Locale::NL_NL, Locale::NL_BE, Locale::DE_DE];

    protected $mailDomain = Mailer::DEALER_MAIL;

    protected $emailType = EmailType::DEALER_REQUEST_DELIVERY_CONFIRMATION;

    protected $query = [
        VehicleType::CAR => [
            'selects' => [
                'vb.id AS lot_id',
                'vb.naam AS last_name',
                'vb.woonplaats AS city',
                'vb.kenteken AS license',
                'makes.merkNaam AS make',
                'models.modelNaam AS model',
                'ddb.bedrijfsnaam as company_name',
                'ddbe.akkoord_email AS email',
                'bc.code AS dealer_country',
                'cc.code AS consumer_country',
                'vl.id AS deliver_id',
                'vl.bedrijvenID AS company_id',
                'lang.code AS language',
            ],
            'table' => ['dealer01_ddmain.verkopen_levering', 'vl'],
            'leftJoin' => [
                ['vl', 'dealer01_ddmain.verzamel_base', 'vb', 'vl.kavelref = vb.id'],
                ['vl', 'dealer01_ddmain.dealerdirect_bedr', 'ddb', 'vl.bedrijvenID = ddb.bedrijvenID'],
                ['vl', 'dealer01_ddmain.dealerdirect_bedr_email', 'ddbe', 'vl.bedrijvenID = ddbe.bedrijvenID'],
                ['vb', 'dealer01_ddmain.countries', 'cc', 'vb.land = cc.country_id'],
                ['ddb', 'dealer01_ddmain.countries', 'bc', 'ddb.land = bc.country_id'],
                ['vb', 'dealer01_webservices.SB_MERKEN', 'makes', 'vb.automerk = makes.merkID'],
                ['vb', 'dealer01_webservices.SB_MODELLEN', 'models', 'vb.autotype = models.modelID'],
                ['ddb', 'dealer01_ddmain.countries', 'lang', 'ddb.pri_lang = lang.country_id']
            ],
            'where' => ['vl.id = :id'],
        ],
        VehicleType::MOTOR => [
            'selects' => [
                'v.kavelref AS lot_id',
                'p.achternaam AS last_name',
                'p.woonplaats AS city',
                'm.kenteken AS license',
                'makes.merkNaam AS make',
                'models.modelNaam AS model',
                'ddb.bedrijfsnaam AS company_name',
                'ddbe.akkoord_email AS email',
                'v.id AS deliver_id',
                'bc.code AS dealer_country',
                'cc.code AS consumer_country',
                'v.bedrijvenID As company_id',
                'lang.code AS language',
            ],
            'table' => ['dealer01_ddmoto.BO_Verkopen_NL', 'v'],
            'leftJoin' => [
                ['p', 'dealer01_ddmain.countries', 'cc', 'p.country_id = cc.country_id'],
                ['ddb', 'dealer01_ddmain.countries', 'bc', 'ddb.land = bc.country_id'],
                ['m', 'dealer01_webservices.SB_MERKEN', 'makes', 'm.merk = makes.merkID'],
                ['m', 'dealer01_webservices.SB_MODELLEN', 'models', 'm.model = models.modelID'],
                ['ddb', 'dealer01_ddmain.countries', 'lang', 'ddb.pri_lang = lang.country_id']
            ],
            'innerJoin' => [
                ['v', 'dealer01_ddmoto.BO_Motorfiets', 'm', 'v.kavelref = m.id'],
                ['m', 'dealer01_ddmoto.BO_Persoon', 'p', 'm.PersoonID = p.PersoonID'],
                ['v', 'dealer01_ddmain.dealerdirect_bedr', 'ddb', 'v.bedrijvenID = ddb.bedrijvenID'],
                ['v', 'dealer01_ddmain.dealerdirect_bedr_email', 'ddbe', 'v.bedrijvenID = ddbe.bedrijvenID'],
            ],
            'where' => ['v.id = :id'],
        ],
    ];


    public function __invoke(int $countryId = Country::NETHERLANDS)
    {
        $this->countryId = $countryId;
        $this->setTable();
        $this->setData();

        $this->substitutions = new Substitutions([
            Key::DEALER_NAME => $this->data['company_name'],
            Key::VEHICLE_MAKE => $this->data['make'],
            Key::VEHICLE_MODEL => $this->data['model'],
            Key::VEHICLE_REGISTRATION_PLATE => $this->data['license'],
            Key::CONSUMER_NAME => $this->data['last_name'],
            Key::CONSUMER_CITY => $this->data['city'],
            Key::URL => $this->getUrl(),
            Key::LOT_NUMBER => $this->data['lot_id'],
        ]);

        // Set vehicleType to car as this mail only exists for car, the content is ok for all vehicleTypes.
        $this->vehicleType = VehicleType::CAR;
        $this->setLocale();
        $this->send();
    }

    /**
     * Overwrites locale if locale is BE, as this mail only exists for NL and DE
     *
     * @return void
     */
    private function setLocale()
    {
        if (!in_array($this->locale, self::ALLOWED_LOCALES, true)) {
            $allowed = implode(', ', self::ALLOWED_LOCALES);

            throw new \Exception("Error, locale '{$this->locale}' is not supported. Allowed locales: '$allowed'");
        }

        if ($this->locale === Locale::NL_BE) {
            $this->locale = Locale::NL_NL;
        }
    }

    /**
     * @return string
     */
    private function getUrl(): string
    {
        return sprintf(
            '%s/purchase-mail/%s',
            $this->getDomain(),
            $this->encryptHash( // Prefix with 0 if vehicleType = motor
                sprintf('%s%s', $this->isMotor ? '0' : '', $this->data['lot_id'])
            )
        );
    }
}
