<?php

namespace Dealerdirect\Fee\Service;

use Dealerdirect\Fee\Repository\FeeRepository;
use Dealerdirect\Fee\Repository\RepositoryException;

/**
 * Class FeeService
 *
 * TODO: Add Locale/country implementation, to switch on fee values
 *
 * @package DealerDirect\Fee
 */
class FeeService
{
    /**
     * @var array Fixed fee mapping for countries other than NL
     */
    private $feeMapping = [
        50000 => 1100,
        40000 => 900,
        30000 => 800,
        20000 => 700,
        10000 => 600,
        7500 => 500,
        5000 => 450,
        4000 => 425,
        3000 => 400,
        2000 => 350,
        1500 => 325,
        1000 => 300,
        750 => 250,
        500 => 190,
        301 => 130,
        201 => 105,
        151 => 90,
        131 => 80,
        101 => 70,
        81 => 65,
        0 => 50
    ];

    /**
     * @var array Fee constants per country, used for fee calculations. For now only NL.
     */
    private $feeConstants;

    /**
     * @var FeeRepository Repository for database transactions
     */
    private $feeRepository = null;

    /**
     * FeeService constructor.
     *
     * @param string $country Country code for fee settings to load. For example 'nl', 'be', ..
     * @param FeeRepository|null $repo FeeRepository for transactional calls. Default null
     *
     * @throws \InvalidArgumentException
     */
    final public function __construct($country, FeeRepository $repo = null)
    {
        $feeCountryConstants = include(__DIR__ . "/../Resource/fees.php");

        if (!array_key_exists($country, $feeCountryConstants)) {
            throw new \InvalidArgumentException("Invalid country: '$country'");
        }

        $this->feeConstants = $feeCountryConstants[$country];

        $this->feeRepository = $repo;
    }

    /**
     * Throws an exception when FeeRepository has been not been initialized (is null)
     *
     * @throws RepositoryException When FeeRepository wasn't initialized for the service
     */
    private function throwExceptionOnNullRepository()
    {
        if (is_null($this->feeRepository)) {
            throw new RepositoryException('FeeRepository not initialized for FeeService');
        }
    }

    /**
     * Get (norm) fee
     *
     * @param integer $bid
     *
     * @return integer fee
     */
    public function getFee($bid)
    {
        $feeStart = $this->feeConstants['norm-start-fee'];
        $modifier = $this->feeConstants['norm-modifier'];
        $bareMinimum = $this->feeConstants['norm-bare-minimum'];

        $fee = $this->calculateFee($bid, $feeStart, $modifier);

        return max(round($fee), $bareMinimum);
    }

    /**
     * @param integer $bid
     *
     * @return integer fee
     */
    public function getMinFee($bid)
    {
        $feeStart = $this->feeConstants['min-start-fee'];
        $modifier = $this->feeConstants['min-modifier'];
        $bareMinimum = $this->feeConstants['min-bare-minimum'];

        $fee = $this->calculateFee($bid, $feeStart, $modifier);

        return max(round($fee), $bareMinimum);
    }

    /**
     * @param integer $bid
     *
     * @return integer fee
     */
    public function getMaxFee($bid)
    {
        $feeStart = $this->feeConstants['max-start-fee'];
        $modifier = $this->feeConstants['max-modifier'];
        $bareMinimum = $this->feeConstants['max-bare-minimum'];

        $fee = $this->calculateFee($bid, $feeStart, $modifier);

        return max(round($fee), $bareMinimum);
    }

    /**
     * @param integer $bid
     * @param integer $feeStart
     * @param integer $modifier
     *
     * @return integer
     */
    private function calculateFee($bid, $feeStart, $modifier)
    {
        return $feeStart * log($bid) - $modifier;
    }

    /**
     * @param integer $fee The fee to check
     * @param integer $minFee minimum fee
     * @param integer $maxFee maximum fee
     *
     * @return bool
     */
    private function checkFee($fee, $minFee, $maxFee)
    {
        return ($fee >= $minFee && $fee <= $maxFee);
    }

    /**
     * Whether the given fee is valid for given bid
     *
     * @param $bid The bid value
     * @param $fee The bid fee
     *
     * @return bool
     */
    public function isValidFee($bid, $fee)
    {
        $minFee = $this->getMinFee($bid);
        $maxFee = $this->getMaxFee($bid);

        return $this->checkFee($fee, $minFee, $maxFee);
    }

    /**
     * Get the handling fee for given lot price
     * (REST-API function for backwards compatibility)
     *
     * @param integer $price
     * @param integer $countryId
     * @return float
     */
    public function getHandlingsFee($price, $countryId)
    {
        if ($countryId == 1) {
            return $this->getFee($price);
        }

        return $this->getFixedMappedFee($price);
    }

    /**
     * Return fee via based on fixed values.
     * Used for countries other than NL
     *
     * @param $bid
     *
     * @return int
     */
    public function getFixedMappedFee($bid)
    {
        $lowestFee = 50;
        foreach ($this->feeMapping as $bidMin => $fee) {
            if ($bid >= $bidMin) {
                return $fee;
            }
            $lowestFee = $fee;
        }

        // default
        return $lowestFee;
    }

    ///
    /// FeeRepository dependant functions
    ///

    /**
     * Returns the current fee for the given (Car) lot
     *
     * @param $lotId
     *
     * @return int
     *
     * @throws RepositoryException
     */
    public function getLotCurrentFee($lotId)
    {
        $this->throwExceptionOnNullRepository();
        return $this->feeRepository->getLotCurrentFee($lotId);
    }

    /**
     * @param int $lotId
     * @param int $lotType
     * @param int $bid
     * @param int $fee
     * @param string $dateSale Lot sale date. Default 'null' for 'not (yet) sold'
     *
     * @return bool
     *
     * @throws RepositoryException
     */
    public function storeFee($lotId, $lotType, $bid, $fee, $dateSale = null)
    {
        $this->throwExceptionOnNullRepository();

        // Get calculated fees:
        $minFee = $this->getMinFee($bid);
        $maxFee = $this->getMaxFee($bid);
        $normFee = $this->getFee($bid);

        return $this->feeRepository->storeFee($lotId, $lotType, $bid, $fee, $normFee, $minFee, $maxFee, $dateSale);
    }
}
