<?php

namespace Dealerdirect\Fee\Repository;

use PDO;

/**
 * Class FeeRepository
 *
 * Repository for transactional Fees handling
 *
 * @package DealerDirect\Fee\Repository
 */
class FeeRepository implements RepositoryInterface
{
    /**
     * @var PDO
     */
    private $pdo;

    /**
     * @param PDO $pdo
     *
     * @throws \InvalidArgumentException
     */
    final public function __construct(PDO $pdo)
    {
        $this->pdo = $pdo;
    }

    /**
     * Returns the current fee for the given (Car) lot from the `verzamel_base`
     *
     * @param $lotId
     *
     * @return int
     */
    public function getLotCurrentFee($lotId)
    {
        $query = $this->pdo->prepare('SELECT com FROM dealer01_ddmain.verzamel_base WHERE id = :lotID');
        $query->execute(array(':lotID' => $lotId));
        $result = $query->fetch();
        if ($result === false) {
            return 0;
        }
        return $result['com'];
    }

    /**
     * Stores Fee calculation entry (mediation fee, norm fee, fee cap, fee floor) into database.
     * This will only be stored for *Cars NL*, based on site reference, in other cases function will return *false*.
     *
     * @param int $lotId
     * @param int $lotType
     * @param string $country
     * @param int $fee
     * @param int $normFee
     * @param int $minFee
     * @param int $maxFee
     * @param string $dateSale
     *
     * @return bool
     */
    public function storeFee($lotId, $lotType, $country, $fee, $normFee, $minFee, $maxFee, $dateSale = null)
    {
        if ($lotType !== 1) {
            return false;
        }

        // TODO: Add company_id and bid value to Table??
        $sql = <<<SQL
INSERT INTO dealer01_ddmain.fees_calculated SET
  lot_id=:lotId,
  lot_type=:lotType,
  date_sale=:dateSale,
  fee_norm=:feeNorm,
  fee_cap=:feeCap,
  fee_floor=:feeFloor,
  fee_mediation=:feeMediation,
  country=:country
ON DUPLICATE KEY UPDATE
  date_sale=:dateSaleUpd,
  fee_norm=:feeNormUpd,
  fee_cap=:feeCapUpd,
  fee_floor=:feeFloorUpd,
  fee_mediation=:feeMediationUpd,
  country=:country
SQL;
        $feeStmt = $this->pdo->prepare($sql);
        return $feeStmt->execute([
            ':lotId' => (int)$lotId,
            ':lotType' => (int)$lotType,
            ':dateSale' => $dateSale,
            ':feeNorm' => (int)$normFee,
            ':feeCap' => (int)$maxFee,
            ':feeFloor' => (int)$minFee,
            ':feeMediation' => (int)$fee,
            ':dateSaleUpd' => $dateSale,
            ':feeNormUpd' => (int)$normFee,
            ':feeCapUpd' => (int)$maxFee,
            ':feeFloorUpd' => (int)$minFee,
            ':feeMediationUpd' => (int)$fee,
            ':country' => $country,
        ]);
    }

    /**
     * Get Lot based country
     *
     * @param int $lotId
     *
     * @return int The country id
     */
    private function getLotCountry($lotId)
    {
        $query = $this->pdo->prepare('SELECT land FROM dealer01_ddmain.verzamel_base WHERE id = :lotID');
        $query->execute(array(':lotID' => $lotId));
        $result = $query->fetch();
        if ($result === false) {
            return 0;
        }
        return (int)$result['land'];
    }
}
