<?php

namespace Dealerdirect\Fee\Service;

use DealerDirect\Generic\Category\VehicleType;
use Exception;

/**
 * Class FeeService
 *
 * @package DealerDirect\Fee
 */
class FeeService
{
    public const SUPPORTED_VEHICLE_TYPES = [VehicleType::CAR, VehicleType::MOTOR, VehicleType::SCOOTER];

    /**
     * @var array
     */
    private $feeMapping;

    /**
     * FeeService constructor.
     *
     * @param string $vehicleType
     * @throws \Exception
     */
    final public function __construct(string $vehicleType)
    {
        if (!file_exists($file = dirname(__DIR__) . "/Resource/{$vehicleType}Mapping.php")) {
            throw new Exception("Invalid vehicleType: {$vehicleType}");
        }

        $this->feeMapping = require $file;
    }

    /**
     * Gets fee by consumer bid
     *
     * @param integer $bid
     * @return int
     */
    public function getFeeByConsumerBid(int $bid): int
    {
        return $this->getFee($bid, true);
    }

    /**
     * Gets fee by dealer bid
     *
     * @param integer $bid
     * @return int
     */
    public function getFeeByDealerBid(int $bid): int
    {
        return $this->getFee($bid, false);
    }

    /**
     * Gets all prices calculated with fee
     *
     * @param integer $bid
     * @param boolean $consumerBid
     * @return array
     */
    public function getPrices(int $bid, bool $consumerBid = false): array
    {
        $fee = $this->getFee($bid, $consumerBid);

        return [
            'fee' => $fee,
            'consumer-price' => $consumerBid ? $bid : max($bid - $fee, 0),
            'dealer-price' => $consumerBid ? ($bid + $fee) : $bid,
        ];
    }

    public function getMinMax($fee, $bid, $consumerBid = false)
    {
        $fixedFee = true;

        return [
            'min' => $fee,
            'max' => $fee,
            'fixed-fee' => $fixedFee,
        ];
    }

    /**
     * Get fee
     *
     * @param integer $bid
     * @param bool $consumerBid
     *
     * @return integer fee
     */
    public function getFee(int $bid, bool $consumerBid = false): int
    {
        // In case the bid is lower than lowest fee, fee becomes the bid.
        if (array_key_first($this->feeMapping) >= $bid) {
            return $bid;
        }


        $possibleFees = array_keys(
            $this->getMatchingFees($this->feeMapping, $bid, $consumerBid)
        );

        return $possibleFees[ceil(array_key_last($possibleFees) / 2)];
    }

    /**
     * Whether the given fee is valid for given bid
     *
     * @param $bid The bid value
     * @param $fee The bid fee
     *
     * @return bool
     */
    public function isValidFee(int $bid, int $fee, $consumerBid = false)
    {
        $minMax = $this->getMinMax($fee, $bid, $consumerBid);

        return $fee === $minMax['min'] || $fee === $minMax['max'];
    }

    /**
     * Filters all fee's that match with given bid.
     *
     * @param array $mapping
     * @param integer $bid
     * @param boolean $consumerBid
     * @return array
     */
    private function getMatchingFees(array $mapping, int $bid, bool $consumerBid): array
    {
        return array_filter($mapping, function ($values, $fee) use ($bid, $consumerBid) {
            $realBid = $consumerBid ? ($bid + $fee) : $bid;

            return $realBid >= $values[0] && $realBid <= $values[1];
        }, ARRAY_FILTER_USE_BOTH);
    }

    /**
     * Get the handling fee for given lot price
     * Only called from dealerdirect.eu
     *
     * @param integer $price
     * @return int
     */
    public function getHandlingsFee($price)
    {
        // In case the bid is lower than lowest fee, fee becomes the bid.
        if (array_key_first($this->feeMapping) >= $price) {
            return $price;
        }

        return $result ?? $this->getFee($price, false);
    }
}
