<?php

namespace Dealerdirect\Fee\Service;

use DealerDirect\Generic\Category\VehicleType;
use Exception;

/**
 * Class FeeService
 *
 * @package DealerDirect\Fee
 */
class FeeService
{
    public const SUPPORTED_VEHICLE_TYPES = [VehicleType::CAR, VehicleType::MOTOR, VehicleType::SCOOTER];

    /**
     * @var array
     */
    private $feeMapping;

    private $euMapping;

    /**
     * FeeService constructor.
     *
     * @param string $vehicleType
     * @throws \Exception
     */
    final public function __construct(string $vehicleType)
    {
        if (!in_array($vehicleType, self::SUPPORTED_VEHICLE_TYPES)) {
            $supported = implode(', ', self::SUPPORTED_VEHICLE_TYPES);
            throw new Exception("Invalid vehicleType '$vehicleType'. Available: $supported");
        }

        $this->feeMapping = include dirname(__DIR__) . "/Resource/{$vehicleType}Mapping.php";
        if (empty($this->feeMapping) || !is_array($this->feeMapping)) {
            throw new Exception("Invalid feeMapping.");
        } elseif (array_key_exists('eu', $this->feeMapping)) {
            $this->euMapping = $this->feeMapping['eu'];
            unset($this->feeMapping['eu']);
        }
    }

    /**
     * Gets fee by consumer bid
     *
     * @param integer $bid
     * @return int
     */
    public function getFeeByConsumerBid(int $bid): int
    {
        return $this->getFee($bid, true);
    }

    /**
     * Gets fee by dealer bid
     *
     * @param integer $bid
     * @return int
     */
    public function getFeeByDealerBid(int $bid): int
    {
        return $this->getFee($bid, false);
    }

    /**
     * Gets all prices calculated with fee
     *
     * @param integer $bid
     * @param boolean $consumerBid
     * @return array
     */
    public function getPrices(int $bid, bool $consumerBid = false): array
    {
        $fee = $this->getFee($bid, $consumerBid);

        return [
            'fee' => $fee,
            'consumer-price' => $consumerBid ? $bid : ($bid - $fee),
            'dealer-price' => $consumerBid ? ($bid + $fee) : $bid,
        ];
    }

    /**
     * Get fee
     *
     * @param integer $bid
     * @param bool $consumerBid
     *
     * @return integer fee
     */
    public function getFee(int $bid, bool $consumerBid = false): int
    {
        $possibleFees = array_keys(
            array_filter($this->feeMapping, function ($values, $fee) use ($bid, $consumerBid) {
                $realBid = $consumerBid ? ($bid + $fee) : $bid;

                return $realBid >= $values[0] && $realBid <= $values[1];
            }, ARRAY_FILTER_USE_BOTH)
        );

        return $possibleFees[ceil(array_key_last($possibleFees) / 2)];
    }

    /**
     * Whether the given fee is valid for given bid
     *
     * @param $bid The bid value
     * @param $fee The bid fee
     *
     * @return bool
     */
    public function isValidFee(int $bid, int $fee, $consumerBid = false)
    {
        $prices = $this->getPrices($bid, $consumerBid);
        $bidCheck = $consumerBid ? $prices['consumer-price'] : $prices['dealer-price'];

        return $prices['fee'] === $fee && $bidCheck === $bid;
    }

    /**
     * Get the handling fee for given lot price
     * Only called from dealerdirect.eu
     *
     * @param integer $price
     * @return int
     */
    public function getHandlingsFee($price)
    {
        if (!empty($this->euMapping)) {
            $euMapping = $this->euMapping;
            $limit = $euMapping[array_key_last($euMapping)][1];
            if ($price <= $limit) {
                return array_key_first(array_filter($euMapping, function ($values) use ($price) {
                    return $price >= $values[0] && $price <= $values[1];
                }, ARRAY_FILTER_USE_BOTH));
            }
        }

        return $this->getFee($price, false);
    }
}
