<?php

namespace Dealerdirect\FinanceApiClient;

use Dealerdirect\FinanceApiClient\Api;
use Dealerdirect\FinanceApiClient\Configuration;

class Client
{
    private $host;
    private $clientId;
    private $clientSecret;
    private $accesstoken;
    private $scope;

    public function __construct(
        string $host,
        string $clientId,
        string $clientSecret,
        array $scope = null
    ) {
        $this->host = $host;
        $this->clientId = $clientId;
        $this->clientSecret = $clientSecret;
        $this->scope = $scope;
    }


    private function getAccessToken(): string
    {
        if (!isset($this->accesstoken)) {
            $client = new \GuzzleHttp\Client();
            $form_params = [
                'grant_type' => 'client_credentials',
                'client_id' => $this->clientId,
                'client_secret' => $this->clientSecret,
            ];
            if (!empty($this->scope)) {
                $form_params['scope'] = join(' ', $this->scope);
            }
            $response = $client->post("{$this->host}/oauth/token", [
                'form_params' => $form_params
            ]);
            $contents = json_decode($response->getBody()->getContents());
            if (empty($contents->access_token)) {
                throw new \Exception('Error, could not get accessToken from exact-api');
            }

            $this->accesstoken =  $contents->access_token;
        }
        return $this->accesstoken;
    }

    public function getInvoicesApi()
    {
        $accesstoken = $this->getAccessToken();

        // Configure OAuth2 access token for authorization: passport
        $config = Configuration::getDefaultConfiguration()
            ->setAccessToken($accesstoken)
            ->setHost($this->host);

        $apiInstance = new Api\InvoicesApi(
            // If you want use custom http client, pass your client which implements `GuzzleHttp\ClientInterface`.
            // This is optional, `GuzzleHttp\Client` will be used as default.
            new \GuzzleHttp\Client(),
            $config
        );
        return $apiInstance;
    }

    public function getDebtorsApi()
    {
        $accesstoken = $this->getAccessToken();

        // Configure OAuth2 access token for authorization: passport
        $config = Configuration::getDefaultConfiguration()
            ->setAccessToken($accesstoken)
            ->setHost($this->host);

        $apiInstance = new Api\DebtorsApi(
            // If you want use custom http client, pass your client which implements `GuzzleHttp\ClientInterface`.
            // This is optional, `GuzzleHttp\Client` will be used as default.
            new \GuzzleHttp\Client(),
            $config
        );
        return $apiInstance;
    }
}
