<?php

declare(strict_types=1);

namespace Dealerdirect\Generic\Enums;

enum Classification: int
{
    case UNKNOWN = 0;
    case YOUNG = 1;
    case DEMOLITION = 11;
    case RECREATION = 21;
    case TRANSPORT = 31;
    case BUDGET = 41;
    case EXPORT = 51;
    case OTHER = 61;
    case PREMIUM = 71;
    case LOW_POTENTIAL = 72;

    public function toLabel(SiteReference $siteReference): ?string
    {
        $defaultLabel = match ($this) {
            self::YOUNG => 'Jong & Fris',
            self::DEMOLITION => 'Schade & Sloop',
            self::RECREATION => 'Recreatie',
            self::TRANSPORT => 'Bestel & Vracht',
            self::BUDGET => 'Budget',
            self::EXPORT => 'Export',
            self::OTHER => 'Overig',
            self::PREMIUM => 'Premium Klassiek',
            self::LOW_POTENTIAL => 'Low Potential',
            default => null,
        };

        // Specific labels for different sites
        return match ($siteReference) {
            SiteReference::MOTORBIKE_DE => match ($this) {
                self::DEMOLITION => 'Schaden und Schrott',
                self::YOUNG => 'Neuwertig',
                self::OTHER => 'Andere',
                default => $defaultLabel,
            },
            SiteReference::BICYCLE_NL => match ($this) {
                self::YOUNG => 'Elektrisch',
                self::DEMOLITION => 'Niet Elektrisch',
                self::TRANSPORT => 'B2B',
                default => $defaultLabel,
            },
            SiteReference::SCOOTER_NL => match ($this) {
                self::YOUNG => 'BS Jong & Fris',
                self::DEMOLITION => 'BS Schade & Sloop',
                default => $defaultLabel,
            },
            default => $defaultLabel,
        };
    }

    public function toCostUnit(SiteReference $siteReference): ?string
    {
        $prefix = $siteReference->toCostUnitPrefix();
        $suffix = match ($siteReference) {
            SiteReference::SCOOTER_NL => match ($this) {
                self::RECREATION, self::TRANSPORT => null,
                default => 'SC', // All scooters get the same cost unit
            },
            SiteReference::CAR_MARKTPLAATS_NL => match ($this) {
                self::LOW_POTENTIAL => null,
                default => 'MP', // All lots from MP will get the same cost unit
            },
            SiteReference::BICYCLE_NL => match ($this) {
                self::YOUNG => 'EL', // Elektrisch
                self::DEMOLITION => 'NE', // Niet Elektrisch
                self::TRANSPORT => 'B2B', // Business to Business
                default => null,
            },
            default => match ($this) {
                self::YOUNG => 'JF', // Jong & Fris
                self::DEMOLITION => 'SS', // Schade & Sloop
                self::RECREATION => 'RC', // Recreatie
                self::TRANSPORT => 'BV', // Bestel & vracht
                self::BUDGET => 'BU', // Budget
                self::EXPORT => 'EX', // Export
                self::OTHER => 'OV', // Overig
                self::PREMIUM => 'PR', // Premium
                default => null,
            },
        };

        return $suffix !== null
            ? "$prefix-$suffix"
            : null;
    }
}
