<?php

declare(strict_types=1);

namespace Dealerdirect\Generic\Enums\Car;

use Dealerdirect\Generic\Enums\Interfaces\CallGroup as CallGroupInterface;
use Dealerdirect\Generic\Enums\PhoneNumber;
use Dealerdirect\Generic\Enums\PhoneNumberType;
use Dealerdirect\Generic\Enums\SiteReference;

/**
 * Contains the call group id for cars from table 'dealer01_ddmain.call_groups_desc'
 */
enum CallGroup: int implements CallGroupInterface
{
    // Car NL
    case CAR_NL_TEAM_1 = 978;
    case CAR_NL_TEAM_2 = 988;
    case CAR_NL_TEAM_3 = 998;
    case CAR_NL_TEAM_4 = 1008;
    case CAR_NL_DEMOLITION = 2;
    case CAR_NL_RECREATION = 3;
    case CAR_NL_7_DAYS = 1062;
    case CAR_NL_14_DAYS = 1082;

    // Car BE
    case CAR_BE_TEAM_1 = 1018;
    case CAR_BE_TEAM_2 = 1032;
    case CAR_BE_TEAM_3 = 1042;
    case CAR_BE_TEAM_4 = 1052;
    case CAR_BE_DEMOLITION = 12;
    case CAR_BE_RECREATION = 965;
    case CAR_BE_7_DAYS = 1072;
    case CAR_BE_14_DAYS = 1092;

    // Marktplaats NL
    case MARKTPLAATS_NL_TEAM_1 = 1028;

    /** @deprecated Use CAR_NL_TEAM_1 instead */
    public const CAR_NL_GROUP_1 = self::CAR_NL_TEAM_1;
    /** @deprecated Use CAR_NL_TEAM_2 instead */
    public const CAR_NL_GROUP_2 = self::CAR_NL_TEAM_2;
    /** @deprecated Use CAR_NL_TEAM_3 instead */
    public const CAR_NL_GROUP_3 = self::CAR_NL_TEAM_3;
    /** @deprecated Use CAR_NL_TEAM_4 instead */
    public const CAR_NL_GROUP_4 = self::CAR_NL_TEAM_4;
    /** @deprecated Use CAR_BE_TEAM_1 instead */
    public const CAR_BE_GROUP_1 = self::CAR_NL_TEAM_1;
    /** @deprecated Use CAR_BE_TEAM_2 instead */
    public const CAR_BE_GROUP_2 = self::CAR_NL_TEAM_2;
    /** @deprecated Use CAR_BE_TEAM_3 instead */
    public const CAR_BE_GROUP_3 = self::CAR_NL_TEAM_3;
    /** @deprecated Use CAR_BE_TEAM_4 instead */
    public const CAR_BE_GROUP_4 = self::CAR_NL_TEAM_4;
    /** @deprecated Use CAR_NL_RECREATION instead */
    public const CAR_NL_GROUP_RECREATION = self::CAR_NL_RECREATION;
    /** @deprecated Use CAR_BE_RECREATION instead */
    public const CAR_BE_GROUP_RECREATION = self::CAR_BE_RECREATION;
    /** @deprecated Use MARKTPLAATS_NL_TEAM_1 instead */
    public const MARKTPLAATS_NL_GROUP_1 = self::MARKTPLAATS_NL_TEAM_1;

    public function toLabel(): string
    {
        return match ($this) {
            self::CAR_NL_TEAM_1,
            self::CAR_BE_TEAM_1,
            self::MARKTPLAATS_NL_TEAM_1 => 'Team 1',

            self::CAR_NL_TEAM_2,
            self::CAR_BE_TEAM_2 => 'Team 2',

            self::CAR_NL_TEAM_3,
            self::CAR_BE_TEAM_3 => 'Team 3',

            self::CAR_NL_TEAM_4,
            self::CAR_BE_TEAM_4 => 'Team 4',

            self::CAR_NL_RECREATION,
            self::CAR_BE_RECREATION => 'Recreatie',

            self::CAR_NL_DEMOLITION,
            self::CAR_BE_DEMOLITION => 'Schade & Sloop',

            self::CAR_NL_7_DAYS,
            self::CAR_BE_7_DAYS => '7 - 14 Dagen',

            self::CAR_NL_14_DAYS,
            self::CAR_BE_14_DAYS => '14+ Dagen',
        };
    }

    public function toPhoneNumber(PhoneNumberType $type): PhoneNumber
    {
        return match ($type) {
            PhoneNumberType::CONSUMER => $this->toConsumerPhoneNumber(),
            PhoneNumberType::DEALER => $this->toDealerPhoneNumber(),
            PhoneNumberType::CUSTOMER_SERVICE => $this->toCustomerServicePhoneNumber(),

            default => throw new \InvalidArgumentException("Invalid PhoneNumberType: $type->value"),
        };
    }

    public function toCustomerServicePhoneNumber(): PhoneNumber
    {
        return match ($this) {
            self::CAR_NL_TEAM_1,
            self::CAR_NL_TEAM_2,
            self::CAR_NL_TEAM_3,
            self::CAR_NL_TEAM_4,
            self::CAR_NL_RECREATION,
            self::CAR_NL_DEMOLITION,
            self::CAR_NL_7_DAYS,
            self::CAR_NL_14_DAYS => PhoneNumber::CAR_NL_CUSTOMER_SERVICE,

            self::CAR_BE_TEAM_1,
            self::CAR_BE_TEAM_2,
            self::CAR_BE_TEAM_3,
            self::CAR_BE_TEAM_4,
            self::CAR_BE_RECREATION,
            self::CAR_BE_DEMOLITION,
            self::CAR_BE_7_DAYS,
            self::CAR_BE_14_DAYS => PhoneNumber::CAR_BE_CUSTOMER_SERVICE,

            self::MARKTPLAATS_NL_TEAM_1 => PhoneNumber::MARKTPLAATS_NL,

            default => throw new \InvalidArgumentException("Invalid CallGroup: $this->value"),
        };
    }

    public function toConsumerPhoneNumber(): PhoneNumber
    {
        return match ($this) {
            self::CAR_NL_TEAM_1 => PhoneNumber::CAR_NL_CONSUMER_1,
            self::CAR_NL_TEAM_2 => PhoneNumber::CAR_NL_CONSUMER_2,
            self::CAR_NL_TEAM_3 => PhoneNumber::CAR_NL_CONSUMER_3,
            self::CAR_NL_TEAM_4,
            self::CAR_NL_RECREATION => PhoneNumber::CAR_NL_CONSUMER_4,
            self::CAR_NL_DEMOLITION => PhoneNumber::CAR_NL_CONSUMER_DEMOLITION,
            self::CAR_NL_7_DAYS,
            self::CAR_NL_14_DAYS => PhoneNumber::CAR_NL_CONSUMER_7_DAYS,

            self::CAR_BE_TEAM_1,
            self::CAR_BE_TEAM_2,
            self::CAR_BE_TEAM_3,
            self::CAR_BE_TEAM_4,
            self::CAR_BE_RECREATION,
            self::CAR_BE_DEMOLITION,
            self::CAR_BE_7_DAYS,
            self::CAR_BE_14_DAYS => PhoneNumber::CAR_BE_CONSUMER_1,

            self::MARKTPLAATS_NL_TEAM_1 => PhoneNumber::MARKTPLAATS_NL,
        };
    }

    public function toDealerPhoneNumber(): PhoneNumber
    {
        return match ($this) {
            self::CAR_NL_TEAM_1 => PhoneNumber::CAR_NL_DEALER_1,
            self::CAR_NL_TEAM_2 => PhoneNumber::CAR_NL_DEALER_2,
            self::CAR_NL_TEAM_3 => PhoneNumber::CAR_NL_DEALER_3,
            self::CAR_NL_TEAM_4,
            self::CAR_NL_RECREATION => PhoneNumber::CAR_NL_DEALER_4,
            self::CAR_NL_DEMOLITION => PhoneNumber::CAR_NL_DEALER_DEMOLITION,
            self::CAR_NL_7_DAYS,
            self::CAR_NL_14_DAYS => PhoneNumber::CAR_NL_DEALER_7_DAYS,

            self::CAR_BE_TEAM_1,
            self::CAR_BE_TEAM_2,
            self::CAR_BE_TEAM_3,
            self::CAR_BE_TEAM_4,
            self::CAR_BE_RECREATION,
            self::CAR_BE_DEMOLITION,
            self::CAR_BE_7_DAYS,
            self::CAR_BE_14_DAYS => PhoneNumber::CAR_BE_DEALER_1,

            self::MARKTPLAATS_NL_TEAM_1 => PhoneNumber::MARKTPLAATS_NL,
        };
    }

    public function toSiteReference(): SiteReference
    {
        return match ($this) {
            self::CAR_NL_TEAM_1,
            self::CAR_NL_TEAM_2,
            self::CAR_NL_TEAM_3,
            self::CAR_NL_TEAM_4,
            self::CAR_NL_RECREATION,
            self::CAR_NL_DEMOLITION,
            self::CAR_NL_7_DAYS,
            self::CAR_NL_14_DAYS => SiteReference::CAR_NL,

            self::CAR_BE_TEAM_1,
            self::CAR_BE_TEAM_2,
            self::CAR_BE_TEAM_3,
            self::CAR_BE_TEAM_4,
            self::CAR_BE_RECREATION,
            self::CAR_BE_DEMOLITION,
            self::CAR_BE_7_DAYS,
            self::CAR_BE_14_DAYS => SiteReference::CAR_BE_NL,

            self::MARKTPLAATS_NL_TEAM_1 => SiteReference::CAR_MARKTPLAATS_NL,
        };
    }
}
