<?php

declare(strict_types=1);

namespace Dealerdirect\Generic\Tests\Enums;

use DealerDirect\Generic\Category\Site;
use Dealerdirect\Generic\Enums\Classification;
use Dealerdirect\Generic\Enums\SiteReference;
use Exception;
use InvalidArgumentException;
use PHPUnit\Framework\Exception as FrameworkException;
use PHPUnit\Framework\ExpectationFailedException;
use PHPUnit\Framework\TestCase;

/**
 * @covers \Dealerdirect\Generic\Enums\Classification
 * @uses \Dealerdirect\Generic\Enums\SiteReference
 */
final class ClassificationTest extends TestCase
{
    /**
     * @dataProvider provide_classification
     */
    public function test_to_label_will_return_string_for_all_site_references(Classification $classification): void
    {
        foreach (SiteReference::cases() as $siteReference) {
            $this->assertIsString($classification->toLabel($siteReference));
        }
    }

    public function test_to_label_will_return_label_with_prefix_for_scooter_nl(): void
    {
        $classifications = [Classification::YOUNG, Classification::DEMOLITION];

        foreach ($classifications as $classification) {
            $this->assertStringStartsWith('BS ', $classification->toLabel(SiteReference::SCOOTER_NL));
        }
    }

    public function test_to_label_will_return_custom_label_for_bicycle_nl(): void
    {
        $checks = [
            'Elektrisch' => Classification::YOUNG,
            'Niet Elektrisch' => Classification::DEMOLITION,
            'B2B' => Classification::TRANSPORT,
        ];

        foreach ($checks as $expected => $classification) {
            $this->assertSame($expected, $classification->toLabel(SiteReference::BICYCLE_NL));
        }
    }

    /**
     * @param Classification[] $invalidClassifications
     *
     * @dataProvider provide_invalid_classifications
     */
    public function test_to_cost_unit_will_return_null_for_invalid_classification(
        SiteReference $siteReference,
        array $invalidClassifications
    ): void {
        foreach ($invalidClassifications as $classification) {
            $this->assertNull($classification->toCostUnit($siteReference));
        }
    }

    /**
     * @dataProvider provide_inactive_site_references
     */
    public function test_to_cost_unit_will_throw_exception_when_site_reference_is_inactive(
        SiteReference $siteReference,
        Classification $classification
    ): void {
        $this->expectException(InvalidArgumentException::class);
        $this->expectExceptionMessage("Cannot get cost unit prefix for inactive SiteReference: $siteReference->value");

        $classification->toCostUnit($siteReference);
    }

    /**
     * @dataProvider provide_expected_cost_units
     */
    public function test_to_cost_unit_will_return_expected_value(
        SiteReference $siteReference,
        array $expectedClassificationCostUnits,
    ): void {
        foreach ($expectedClassificationCostUnits as $expectedCostUnit => $classification) {
            $costUnit = $classification->toCostUnit($siteReference);
            $this->assertIsString($costUnit);
            $this->assertSame($expectedCostUnit, $costUnit);
        }
    }

    public static function provide_expected_cost_units(): iterable
    {
        return [
            'car_nl' => [
                SiteReference::CAR_NL,
                [
                    'ANL-JF' => Classification::YOUNG,
                    'ANL-SS' => Classification::DEMOLITION,
                    'ANL-RC' => Classification::RECREATION,
                    'ANL-BV' => Classification::TRANSPORT,
                    'ANL-BU' => Classification::BUDGET,
                    'ANL-EX' => Classification::EXPORT,
                    'ANL-OV' => Classification::OTHER,
                    'ANL-PR' => Classification::PREMIUM,
                ]
            ],
            'car_be' => [
                SiteReference::CAR_BE_NL,
                [
                    'ABE-JF' => Classification::YOUNG,
                    'ABE-SS' => Classification::DEMOLITION,
                    'ABE-RC' => Classification::RECREATION,
                    'ABE-BV' => Classification::TRANSPORT,
                    'ABE-BU' => Classification::BUDGET,
                    'ABE-EX' => Classification::EXPORT,
                    'ABE-OV' => Classification::OTHER,
                    'ABE-PR' => Classification::PREMIUM,
                ]
            ],
            'car_marktplaats_nl' => [
                SiteReference::CAR_MARKTPLAATS_NL,
                [
                    'ANL-MP' => Classification::YOUNG,
                    'ANL-MP' => Classification::DEMOLITION,
                    'ANL-MP' => Classification::RECREATION,
                    'ANL-MP' => Classification::TRANSPORT,
                    'ANL-MP' => Classification::BUDGET,
                    'ANL-MP' => Classification::EXPORT,
                    'ANL-MP' => Classification::OTHER,
                    'ANL-MP' => Classification::PREMIUM,
                ]
            ],
            'scooter_nl' => [
                SiteReference::SCOOTER_NL,
                [
                    'MNL-SC' => Classification::YOUNG,
                    'MNL-SC' => Classification::DEMOLITION,
                    'MNL-SC' => Classification::LOW_POTENTIAL,
                    'MNL-SC' => Classification::OTHER,
                ]
            ],
            'bicycle_nl' => [
                SiteReference::BICYCLE_NL,
                [
                    'BNL-EL' => Classification::YOUNG,
                    'BNL-NE' => Classification::DEMOLITION,
                    'BNL-B2B' => Classification::TRANSPORT,
                ]
            ],
            'motor_nl' => [
                SiteReference::MOTORBIKE_NL,
                [
                    'MNL-JF' => Classification::YOUNG,
                    'MNL-OV' => Classification::OTHER,
                    'MNL-BU' => Classification::BUDGET,
                    'MNL-SS' => Classification::DEMOLITION,
                    'MNL-PR' => Classification::PREMIUM,
                ]
            ],
            'motor_be' => [
                SiteReference::MOTORBIKE_BE,
                [
                    'MBE-JF' => Classification::YOUNG,
                    'MBE-OV' => Classification::OTHER,
                    'MBE-BU' => Classification::BUDGET,
                    'MBE-SS' => Classification::DEMOLITION,
                    'MBE-PR' => Classification::PREMIUM,
                ]
            ],
            'motor_de' => [
                SiteReference::MOTORBIKE_DE,
                [
                    'MDE-JF' => Classification::YOUNG,
                    'MDE-OV' => Classification::OTHER,
                    'MDE-BU' => Classification::BUDGET,
                    'MDE-SS' => Classification::DEMOLITION,
                    'MDE-PR' => Classification::PREMIUM,
                ]
            ],
        ];
    }

    public static function provide_classification(): iterable
    {
        foreach (Classification::cases() as $case) {
            if ($case !== Classification::UNKNOWN) {
                yield $case->name => [$case];
            }
        }
    }

    public static function provide_invalid_classifications(): array
    {
        return [
            'car_nl' => [SiteReference::CAR_NL, [Classification::LOW_POTENTIAL]],
            'car_be_nl' => [SiteReference::CAR_BE_NL, [Classification::LOW_POTENTIAL]],
            // 'car_de' => [SiteReference::CAR_DE, Classification::cases()],
            'car_marktplaats_nl' => [SiteReference::CAR_MARKTPLAATS_NL, [Classification::LOW_POTENTIAL]],
            // 'car_be_fr' => [SiteReference::CAR_BE_FR, Classification::cases()],
            'motorbike_nl' => [SiteReference::MOTORBIKE_NL, [Classification::LOW_POTENTIAL]],
            'motorbike_be' => [SiteReference::MOTORBIKE_BE, [Classification::LOW_POTENTIAL]],
            // 'motorbike_be_legacy' => [SiteReference::MOTORBIKE_BE_LEGACY, Classification::cases()],
            'motorbike_de' => [SiteReference::MOTORBIKE_DE, [Classification::LOW_POTENTIAL]],
            'bicycle_nl' => [
                SiteReference::BICYCLE_NL,
                array_filter(
                    Classification::cases(),
                    fn (Classification $classification) => !in_array($classification, [
                        Classification::YOUNG,
                        Classification::DEMOLITION,
                        Classification::TRANSPORT,
                    ], true)
                )
            ],
            'scooter_nl' => [
                SiteReference::SCOOTER_NL,
                [Classification::RECREATION, Classification::TRANSPORT],
                // array_filter(
                //     Classification::cases(),
                //     fn (Classification $classification) => !in_array($classification, [
                //         Classification::YOUNG,
                //         Classification::DEMOLITION,
                //     ], true)
                // )
            ],
        ];
    }

    public static function provide_inactive_site_references(): array
    {
        return [
            [SiteReference::CAR_BE_FR, Classification::YOUNG],
            [SiteReference::CAR_DE, Classification::DEMOLITION],
            [SiteReference::MOTORBIKE_BE_LEGACY, Classification::TRANSPORT],
            [SiteReference::UNKNOWN, Classification::BUDGET],
        ];
    }
}
