<?php

declare(strict_types=1);

namespace Dealerdirect\Generic\Tests\Enums;

use BackedEnum;
use Dealerdirect\Generic\Enums;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\DataProvider;
use PHPUnit\Framework\Attributes\DoesNotPerformAssertions;
use PHPUnit\Framework\TestCase;
use ReflectionClass;
use ReflectionMethod;
use UnhandledMatchError;
use UnitEnum;

#[CoversClass(Enums\Car\CallGroup::class)]
#[CoversClass(Enums\Motorbike\CallGroup::class)]
#[CoversClass(Enums\SiteReference::class)]
#[CoversClass(Enums\VehicleType::class)]
#[CoversClass(Enums\Classification::class)]
#[CoversClass(Enums\PhoneNumberType::class)]
#[CoversClass(Enums\PhoneNumber::class)]
#[CoversClass(Enums\Country::class)]
#[CoversClass(Enums\CallStatus::class)]
final class GeneralEnumTest extends TestCase
{
    private static array $enums = [
        Enums\SiteReference::class,
        Enums\VehicleType::class,
        Enums\Classification::class,
        Enums\PhoneNumberType::class,
        Enums\PhoneNumber::class,
        Enums\Country::class,
        Enums\Car\CallGroup::class,
        Enums\Motorbike\CallGroup::class,
        Enums\CallStatus::class,
    ];

    public function test_all_enums_are_loaded(): void
    {
        $enumsFolder = dirname(__DIR__, 2) . '/src/Enums';
        $recursiveIterator = new \RecursiveIteratorIterator(
            new \RecursiveDirectoryIterator($enumsFolder)
        );

        /** @var \SplFileInfo $file */
        foreach ($recursiveIterator as $file) {
            if (!$file->isFile() || 'php' !== $file->getExtension() || str_contains('Interfaces', $file->getPath())) {
                continue;
            }

            $namespacePrefix = 'Dealerdirect\Generic\Enums';
            $namespaceSuffix = str_replace([$enumsFolder, '.php'], '', $file->getPathname());
            $enumNamespace = $namespacePrefix . str_replace('/', '\\', $namespaceSuffix);

            if (!class_exists($enumNamespace)) {
                continue;
            }

            $reflection = new ReflectionClass($enumNamespace);
            if ($reflection->isEnum()) {
                $this->assertContains($enumNamespace, self::$enums);
            }
        }
    }

    #[DataProvider('provide_enums')]
    #[DoesNotPerformAssertions]
    public function test_all_methods(UnitEnum|BackedEnum $enum): void
    {
        // $this->doesNotPerformAssertions();

        $reflection = new ReflectionClass($enum);
        $enumClass = $reflection->getName();

        foreach ($reflection->getMethods(ReflectionMethod::IS_PUBLIC) as $method) {
            if ($method->isStatic() || (
                'toCostUnitPrefix' === $method->getName()
                && $enum instanceof Enums\SiteReference
                && !$enum->isActive()
            )) {
                continue;
            }

            $args = [];
            foreach ($method->getParameters() as $parameter) {
                $parameterType = $parameter->getType()?->getName();
                if (in_array($parameterType, self::$enums, true)) {
                    $parameterCases = Enums\SiteReference::class === $parameterType
                        ? Enums\SiteReference::activeCases()
                        : $parameterType::cases();

                    $args[] = $parameterCases[array_rand($parameterCases)];

                    continue;
                }

                $this->fail("Unhandled parameter type: {$parameterType}");
            }

            try {
                $method->invoke($enum, ...$args);
            } catch (UnhandledMatchError $e) {
                $this->fail(
                    "UnhandledMatchError: {$enumClass}::{$enum->name}->{$method->getName()}()"
                );
            }
        }
    }

    public static function provide_enums(): iterable
    {
        foreach (self::$enums as $enumClass) {
            foreach ($enumClass::cases() as $case) {
                yield "{$enumClass}::{$case->name}" => [$case];
            }
        }
    }
}
