<?php

declare(strict_types=1);

namespace Dealerdirect\Generic\Enums;

use Dealerdirect\Generic\Enums\Interfaces\CallGroup;

enum PhoneNumber: string
{
    // Car BE
    case CAR_BE_CUSTOMER_SERVICE = '+3216980382';
    case CAR_BE_CONSUMER_1 = '+3216980389';
    case CAR_BE_DEALER_1 = '+3216980381';

    // Car NL
    case CAR_NL_CUSTOMER_SERVICE = '+31532084446';
    case CAR_NL_CONSUMER_1 = '+31537600944';
    case CAR_NL_CONSUMER_2 = '+31537600945';
    case CAR_NL_CONSUMER_3 = '+31537600946';
    case CAR_NL_CONSUMER_4 = '+31537600947';
    case CAR_NL_DEALER_1 = '+31537600940';
    case CAR_NL_DEALER_2 = '+31537600941';
    case CAR_NL_DEALER_3 = '+31537600942';
    case CAR_NL_DEALER_4 = '+31537600943';

    // Marktplaats NL
    case MARKTPLAATS_NL = '+31857733443';

    // Motorbike NL & Scooter NL
    case MOTOR_NL_CUSTOMER_SERVICE = '+31532084449';
    case MOTOR_NL_CONSUMER_1 = '+31537600030';
    case MOTOR_NL_DEALER_1 = '+31537600029';

    // Motorbike BE
    // TODO: Check MOTOR_BE_CUSTOMER_SERVICE, as the legacy code is using the same number as MOTOR_NL_CUSTOMER_SERVICE
    // case MOTOR_BE_CUSTOMER_SERVICE = '+3216980332';
    case MOTOR_BE_CONSUMER_1 = '+3216980331';
    case MOTOR_BE_DEALER_1 = '+3216980341';

    // Motorbike DE
    case MOTOR_DE_CUSTOMER_SERVICE = '+4925629993012';
    case MOTOR_DE_CONSUMER_1 = '+4925629993010';
    case MOTOR_DE_DEALER_1 = '+4925629993011';

    // Bicycle NL
    case BICYCLE_NL_CUSTOMER_SERVICE = '+31532084443';
    case BICYCLE_NL_CONSUMER_1 = '+31532084444';
    case BICYCLE_NL_DEALER_1 = '+31532084442';

    public function toBrand(): string
    {
        return match (true) {
            $this === self::MARKTPLAATS_NL,
            $this->isConsumerPhone(),
            $this->isCustomerServicePhone() => $this->toSiteReference()->toLabel(),

            $this->isDealerPhone() => 'Dealerdirect',
        };
    }

    public function toSiteReference(): SiteReference
    {
        return match ($this) {
            self::CAR_NL_CONSUMER_1,
            self::CAR_NL_CONSUMER_2,
            self::CAR_NL_CONSUMER_3,
            self::CAR_NL_CONSUMER_4,
            self::CAR_NL_DEALER_1,
            self::CAR_NL_DEALER_2,
            self::CAR_NL_DEALER_3,
            self::CAR_NL_DEALER_4,
            self::CAR_NL_CUSTOMER_SERVICE => SiteReference::CAR_NL,

            self::CAR_BE_CONSUMER_1,
            self::CAR_BE_DEALER_1,
            self::CAR_BE_CUSTOMER_SERVICE => SiteReference::CAR_BE_NL,

            self::MARKTPLAATS_NL => SiteReference::CAR_MARKTPLAATS_NL,

            self::MOTOR_NL_CONSUMER_1,
            self::MOTOR_NL_DEALER_1,
            self::MOTOR_NL_CUSTOMER_SERVICE => SiteReference::MOTORBIKE_NL,

            self::MOTOR_BE_CONSUMER_1,
            self::MOTOR_BE_DEALER_1 => SiteReference::MOTORBIKE_BE,

            self::MOTOR_DE_CONSUMER_1,
            self::MOTOR_DE_DEALER_1,
            self::MOTOR_DE_CUSTOMER_SERVICE => SiteReference::MOTORBIKE_DE,

            self::BICYCLE_NL_CONSUMER_1,
            self::BICYCLE_NL_DEALER_1,
            self::BICYCLE_NL_CUSTOMER_SERVICE => SiteReference::BICYCLE_NL,
        };
    }

    /**
     * Gets the related CallGroup enum for this PhoneNumber.
     *
     * @return CallGroup[]
     */
    public function toCallGroups(): array
    {
        return match ($this) {
            self::CAR_BE_CONSUMER_1, self::CAR_BE_DEALER_1, self::CAR_BE_CUSTOMER_SERVICE => [
                Car\CallGroup::CAR_BE_GROUP_1,
            ],
            self::CAR_NL_CUSTOMER_SERVICE => [
                Car\CallGroup::CAR_NL_GROUP_1,
                Car\CallGroup::CAR_NL_GROUP_2,
                Car\CallGroup::CAR_NL_GROUP_3,
                Car\CallGroup::CAR_NL_GROUP_4,
            ],
            self::CAR_NL_CONSUMER_1, self::CAR_NL_DEALER_1 => [
                Car\CallGroup::CAR_NL_GROUP_1,
            ],
            self::CAR_NL_CONSUMER_2, self::CAR_NL_DEALER_2 => [
                Car\CallGroup::CAR_NL_GROUP_2,
            ],
            self::CAR_NL_CONSUMER_3, self::CAR_NL_DEALER_3 => [
                Car\CallGroup::CAR_NL_GROUP_3,
            ],
            self::CAR_NL_CONSUMER_4, self::CAR_NL_DEALER_4 => [
                Car\CallGroup::CAR_NL_GROUP_4,
            ],
            self::MARKTPLAATS_NL => [
                Car\CallGroup::MARKTPLAATS_NL_GROUP_1,
            ],
            self::MOTOR_NL_CONSUMER_1, self::MOTOR_NL_DEALER_1, self::MOTOR_NL_CUSTOMER_SERVICE => [
                Motorbike\CallGroup::MOTOR_NL_GROUP_1,
                Motorbike\CallGroup::MOTOR_NL_GROUP_2,
                Motorbike\CallGroup::MOTOR_NL_GROUP_3,
                Motorbike\CallGroup::MOTOR_NL_GROUP_4,
                Motorbike\CallGroup::SCOOTER_NL_OTHER,
                Motorbike\CallGroup::SCOOTER_NL_YOUNG_FRESH,
                Motorbike\CallGroup::SCOOTER_NL_LOW_POTENTIAL,
            ],
            self::MOTOR_BE_CONSUMER_1, self::MOTOR_BE_DEALER_1 => [
                Motorbike\CallGroup::MOTOR_BE_GROUP_1,
                Motorbike\CallGroup::MOTOR_BE_GROUP_2,
                Motorbike\CallGroup::MOTOR_BE_GROUP_3,
                Motorbike\CallGroup::MOTOR_BE_GROUP_4,
            ],
            self::MOTOR_DE_CONSUMER_1, self::MOTOR_DE_DEALER_1, self::MOTOR_DE_CUSTOMER_SERVICE => [
                Motorbike\CallGroup::MOTOR_DE_GROUP_1,
                Motorbike\CallGroup::MOTOR_DE_GROUP_2,
            ],
            self::BICYCLE_NL_CONSUMER_1, self::BICYCLE_NL_DEALER_1, self::BICYCLE_NL_CUSTOMER_SERVICE => [
                Motorbike\CallGroup::BICYCLE_NL_ELECTRIC,
                Motorbike\CallGroup::BICYCLE_NL_NON_ELECTRIC,
                Motorbike\CallGroup::BICYCLE_NL_B2B,
            ],
        };
    }

    public function toPhoneNumberTypes(): array
    {
        return match ($this) {
            self::CAR_NL_CONSUMER_1,
            self::CAR_NL_CONSUMER_2,
            self::CAR_NL_CONSUMER_3,
            self::CAR_NL_CONSUMER_4,
            self::CAR_BE_CONSUMER_1,
            self::MOTOR_NL_CONSUMER_1,
            self::MOTOR_BE_CONSUMER_1,
            self::MOTOR_DE_CONSUMER_1,
            self::BICYCLE_NL_CONSUMER_1 => [PhoneNumberType::CONSUMER],

            self::CAR_NL_DEALER_1,
            self::CAR_NL_DEALER_2,
            self::CAR_NL_DEALER_3,
            self::CAR_NL_DEALER_4,
            self::CAR_BE_DEALER_1,
            self::MOTOR_NL_DEALER_1,
            self::MOTOR_BE_DEALER_1,
            self::MOTOR_DE_DEALER_1,
            self::BICYCLE_NL_DEALER_1 => [PhoneNumberType::DEALER],

            self::CAR_NL_CUSTOMER_SERVICE,
            self::CAR_BE_CUSTOMER_SERVICE,
            self::MOTOR_NL_CUSTOMER_SERVICE,
            // self::MOTOR_BE_CUSTOMER_SERVICE,
            self::MOTOR_DE_CUSTOMER_SERVICE,
            self::BICYCLE_NL_CUSTOMER_SERVICE => [PhoneNumberType::CUSTOMER_SERVICE],

            self::MARKTPLAATS_NL => [
                PhoneNumberType::CONSUMER,
                PhoneNumberType::DEALER,
                PhoneNumberType::CUSTOMER_SERVICE,
            ],
        };
    }

    public function isDealerPhone(): bool
    {
        return in_array(PhoneNumberType::DEALER, $this->toPhoneNumberTypes(), true);
    }

    public function isConsumerPhone(): bool
    {
        return in_array(PhoneNumberType::CONSUMER, $this->toPhoneNumberTypes(), true);
    }

    public function isCustomerServicePhone(): bool
    {
        return in_array(PhoneNumberType::CUSTOMER_SERVICE, $this->toPhoneNumberTypes(), true);
    }
}
