<?php

declare(strict_types=1);

namespace Dealerdirect\Generic\Enums;

enum VehicleType: string
{
    case CAR = 'car';
    case MOTORBIKE = 'motorbike';
    case SCOOTER = 'scooter';
    case BICYCLE = 'bicycle';
    case CAMPER = 'camper';
    case CARAVAN = 'caravan';

    /**
     * @return SiteReference[]
     */
    public function toSiteReferences(): array
    {
        return match ($this) {
            self::CAR, self::CAMPER, self::CARAVAN => [
                SiteReference::CAR_NL,
                SiteReference::CAR_BE_NL,
                SiteReference::CAR_DE,
                SiteReference::CAR_MARKTPLAATS_NL,
                SiteReference::CAR_BE_FR,
            ],
            self::MOTORBIKE => [
                SiteReference::MOTORBIKE_NL,
                SiteReference::MOTORBIKE_BE,
                SiteReference::MOTORBIKE_DE,
            ],
            self::SCOOTER => [
                SiteReference::SCOOTER_NL,
            ],
            self::BICYCLE => [
                SiteReference::BICYCLE_NL,
            ],
        };
    }

    public function isCarBased(): bool
    {
        return in_array($this, [self::CAR, self::CAMPER, self::CARAVAN], true);
    }

    public function isMotorbikeBased(): bool
    {
        return in_array($this, [self::MOTORBIKE, self::SCOOTER, self::BICYCLE], true);
    }

    public function isRecreation(): bool
    {
        return in_array($this, [self::CAMPER, self::CARAVAN], true);
    }

    public function toBaseType(): self
    {
        return match ($this) {
            self::CAMPER,
            self::CARAVAN,
            self::CAR => self::CAR,

            self::SCOOTER,
            self::BICYCLE,
            self::MOTORBIKE => self::MOTORBIKE,
        };
    }

    public static function fromLegacy(string $vehicleType): self
    {
        $vehicleType = match ($vehicleType) {
            'motobike' => self::MOTORBIKE->value,
            default => $vehicleType,
        };

        return self::from($vehicleType);
    }
}
