<?php

declare(strict_types=1);

namespace Dealerdirect\Generic\Enums;

use DealerDirect\Generic\Category\Locale;

enum SiteReference: int
{
    // Car
    case UNKNOWN = 0;
    case CAR_NL = 1;
    case CAR_BE_NL = 2;
    case CAR_DE = 3;
    case CAR_MARKTPLAATS_NL = 7;
    case CAR_BE_FR = 200;

    // Motorbike
    case MOTORBIKE_NL = 4;
    case SCOOTER_NL = 5;
    case SCOOTER_DE_LEGACY = 500;
    case BICYCLE_NL = 6;
    case MOTORBIKE_BE = 40;
    case MOTORBIKE_BE_LEGACY = 50;
    case MOTORBIKE_DE = 400;

    public function toLabel(): string
    {
        return match ($this) {
            self::UNKNOWN, self::CAR_NL => 'Ikwilvanmijnautoaf',
            self::CAR_BE_NL => 'Ikwilvanmijnautoaf',
            self::CAR_BE_FR => 'Jevoudraisvendremavoiture',
            self::CAR_DE => 'Ichwillmeinautoloswerden',
            self::CAR_MARKTPLAATS_NL => 'Marktplaats',
            self::MOTORBIKE_NL => 'Ikwilvanmijnmotoraf',
            self::SCOOTER_NL, self::SCOOTER_DE_LEGACY => 'Ikwilvanmijnscooteraf',
            self::BICYCLE_NL => 'Ikwilvanmijnfietsaf',
            self::MOTORBIKE_BE, self::MOTORBIKE_BE_LEGACY => 'Ikwilvanmijnmotoraf',
            self::MOTORBIKE_DE => 'Ichwillmeinmotorradloswerden',
        };
    }

    public function isActive(): bool
    {
        return in_array($this, self::activeCases(), true);
    }

    /**
     * @return CallGroup[]
     */
    public function toCallGroups(): array
    {
        return match ($this) {
            self::CAR_NL => [
                Car\CallGroup::CAR_NL_GROUP_1,
                Car\CallGroup::CAR_NL_GROUP_2,
                Car\CallGroup::CAR_NL_GROUP_3,
                Car\CallGroup::CAR_NL_GROUP_4,
                Car\CallGroup::CAR_NL_GROUP_RECREATION,
            ],
            self::CAR_BE_NL => [
                Car\CallGroup::CAR_BE_GROUP_1,
                Car\CallGroup::CAR_BE_GROUP_2,
                Car\CallGroup::CAR_BE_GROUP_3,
                Car\CallGroup::CAR_BE_GROUP_4,
                Car\CallGroup::CAR_BE_GROUP_RECREATION,
            ],
            self::CAR_MARKTPLAATS_NL => [
                Car\CallGroup::MARKTPLAATS_NL_GROUP_1,
            ],
            self::MOTORBIKE_NL => [
                Motorbike\CallGroup::MOTOR_NL_GROUP_1,
                Motorbike\CallGroup::MOTOR_NL_GROUP_2,
                Motorbike\CallGroup::MOTOR_NL_GROUP_3,
                Motorbike\CallGroup::MOTOR_NL_GROUP_4,
            ],
            self::SCOOTER_NL => [
                Motorbike\CallGroup::SCOOTER_NL_OTHER,
                Motorbike\CallGroup::SCOOTER_NL_YOUNG_FRESH,
                Motorbike\CallGroup::SCOOTER_NL_LOW_POTENTIAL,
            ],
            self::BICYCLE_NL => [
                Motorbike\CallGroup::BICYCLE_NL_ELECTRIC,
                Motorbike\CallGroup::BICYCLE_NL_NON_ELECTRIC,
                Motorbike\CallGroup::BICYCLE_NL_B2B,
            ],
            self::MOTORBIKE_BE => [
                Motorbike\CallGroup::MOTOR_BE_GROUP_1,
                Motorbike\CallGroup::MOTOR_BE_GROUP_2,
                Motorbike\CallGroup::MOTOR_BE_GROUP_3,
                Motorbike\CallGroup::MOTOR_BE_GROUP_4,
            ],
            self::MOTORBIKE_DE => [
                Motorbike\CallGroup::MOTOR_DE_GROUP_1,
                Motorbike\CallGroup::MOTOR_DE_GROUP_2,
            ],
            default => [],
        };
    }

    public function toVehicleType(): VehicleType
    {
        return match ($this) {
            self::UNKNOWN,
            self::CAR_NL,
            self::CAR_BE_NL,
            self::CAR_DE,
            self::CAR_MARKTPLAATS_NL,
            self::CAR_BE_FR => VehicleType::CAR,

            self::MOTORBIKE_NL,
            self::MOTORBIKE_BE,
            self::MOTORBIKE_BE_LEGACY,
            self::MOTORBIKE_DE => VehicleType::MOTORBIKE,

            self::SCOOTER_NL,
            self::SCOOTER_DE_LEGACY => VehicleType::SCOOTER,

            self::BICYCLE_NL => VehicleType::BICYCLE,
        };
    }

    public function toLegacyLocale(): string
    {
        return match ($this) {
            self::UNKNOWN,
            self::CAR_NL,
            self::CAR_MARKTPLAATS_NL,
            self::MOTORBIKE_NL,
            self::SCOOTER_NL,
            self::BICYCLE_NL => Locale::NL_NL,

            self::CAR_BE_NL,
            self::MOTORBIKE_BE,
            self::MOTORBIKE_BE_LEGACY => Locale::NL_BE,

            self::CAR_DE,
            self::MOTORBIKE_DE,
            self::SCOOTER_DE_LEGACY => Locale::DE_DE,
            self::CAR_BE_FR => Locale::FR_BE,
        };
    }

    public function toCostUnitPrefix(): string
    {
        if (!$this->isActive()) {
            throw new \InvalidArgumentException("Cannot get cost unit prefix for inactive SiteReference: $this->value");
        }

        return match ($this) {
            self::UNKNOWN,
            self::CAR_NL,
            self::CAR_MARKTPLAATS_NL => 'ANL',
            self::MOTORBIKE_NL,
            self::SCOOTER_NL => 'MNL',
            self::BICYCLE_NL => 'BNL',
            self::CAR_BE_NL => 'ABE',
            self::MOTORBIKE_BE => 'MBE',
            self::MOTORBIKE_DE => 'MDE',

            default => throw new \InvalidArgumentException(
                "Could not find cost unit prefix for SiteReference: $this->value"
            ),
        };
    }

    public function toCountry(): Country
    {
        return match ($this) {
            self::UNKNOWN,
            self::CAR_NL,
            self::CAR_MARKTPLAATS_NL,
            self::MOTORBIKE_NL,
            self::SCOOTER_NL,
            self::BICYCLE_NL => Country::NL,

            self::CAR_BE_NL,
            self::CAR_BE_FR,
            self::MOTORBIKE_BE,
            self::MOTORBIKE_BE_LEGACY => Country::BE,

            self::MOTORBIKE_DE,
            self::SCOOTER_DE_LEGACY,
            self::CAR_DE => Country::DE,
        };
    }

    /**
     * @return self[]
     */
    public static function activeCases(): array
    {
        return [
            self::CAR_NL,
            self::CAR_BE_NL,
            self::CAR_MARKTPLAATS_NL,
            self::MOTORBIKE_NL,
            self::SCOOTER_NL,
            self::BICYCLE_NL,
            self::MOTORBIKE_BE,
            self::MOTORBIKE_DE,
        ];
    }
}
