<?php

declare(strict_types=1);

namespace Dealerdirect\Generic\Enums\Car;

use Dealerdirect\Generic\Enums\Interfaces\CallGroup as CallGroupInterface;
use Dealerdirect\Generic\Enums\PhoneNumber;
use Dealerdirect\Generic\Enums\PhoneNumberType;
use Dealerdirect\Generic\Enums\SiteReference;

/**
 * Contains the call group id for cars from table 'dealer01_ddmain.call_groups_desc'
 */
enum CallGroup: int implements CallGroupInterface
{
    // Car NL
    case CAR_NL_GROUP_1 = 978;
    case CAR_NL_GROUP_2 = 988;
    case CAR_NL_GROUP_3 = 998;
    case CAR_NL_GROUP_4 = 1008;

    // Car BE
    case CAR_BE_GROUP_1 = 1018;
    case CAR_BE_GROUP_2 = 1032;
    case CAR_BE_GROUP_3 = 1042;
    case CAR_BE_GROUP_4 = 1052;

    // Marktplaats NL
    case MARKTPLAATS_NL_GROUP_1 = 1028;

    public function toLabel(): string
    {
        return match ($this) {
            self::CAR_NL_GROUP_1,
            self::CAR_BE_GROUP_1,
            self::MARKTPLAATS_NL_GROUP_1 => 'Groep 1',

            self::CAR_NL_GROUP_2,
            self::CAR_BE_GROUP_2 => 'Groep 2',

            self::CAR_NL_GROUP_3,
            self::CAR_BE_GROUP_3 => 'Groep 3',

            self::CAR_NL_GROUP_4,
            self::CAR_BE_GROUP_4 => 'Groep 4',
        };
    }

    public function toPhoneNumber(PhoneNumberType $type): PhoneNumber
    {
        return match ($type) {
            PhoneNumberType::CONSUMER => $this->toConsumerPhoneNumber(),
            PhoneNumberType::DEALER => $this->toDealerPhoneNumber(),
            PhoneNumberType::CUSTOMER_SERVICE => $this->toCustomerServicePhoneNumber(),

            default => throw new \InvalidArgumentException("Invalid PhoneNumberType: $type->value"),
        };
    }

    public function toCustomerServicePhoneNumber(): PhoneNumber
    {
        return match ($this) {
            self::CAR_NL_GROUP_1,
            self::CAR_NL_GROUP_2,
            self::CAR_NL_GROUP_3,
            self::CAR_NL_GROUP_4 => PhoneNumber::CAR_NL_CUSTOMER_SERVICE,
            self::CAR_BE_GROUP_1,
            self::CAR_BE_GROUP_2,
            self::CAR_BE_GROUP_3,
            self::CAR_BE_GROUP_4 => PhoneNumber::CAR_BE_CUSTOMER_SERVICE,
            self::MARKTPLAATS_NL_GROUP_1 => PhoneNumber::MARKTPLAATS_NL,

            default => throw new \InvalidArgumentException("Invalid CallGroup: $this->value"),
        };
    }

    public function toConsumerPhoneNumber(): PhoneNumber
    {
        return match ($this) {
            self::CAR_NL_GROUP_1 => PhoneNumber::CAR_NL_CONSUMER_1,
            self::CAR_NL_GROUP_2 => PhoneNumber::CAR_NL_CONSUMER_2,
            self::CAR_NL_GROUP_3 => PhoneNumber::CAR_NL_CONSUMER_3,
            self::CAR_NL_GROUP_4 => PhoneNumber::CAR_NL_CONSUMER_4,
            self::CAR_BE_GROUP_1 => PhoneNumber::CAR_BE_CONSUMER_1,
            self::CAR_BE_GROUP_2 => PhoneNumber::CAR_BE_CONSUMER_1,
            self::CAR_BE_GROUP_3 => PhoneNumber::CAR_BE_CONSUMER_1,
            self::CAR_BE_GROUP_4 => PhoneNumber::CAR_BE_CONSUMER_1,
            self::MARKTPLAATS_NL_GROUP_1 => PhoneNumber::MARKTPLAATS_NL,
        };
    }

    public function toDealerPhoneNumber(): PhoneNumber
    {
        return match ($this) {
            self::CAR_NL_GROUP_1 => PhoneNumber::CAR_NL_DEALER_1,
            self::CAR_NL_GROUP_2 => PhoneNumber::CAR_NL_DEALER_2,
            self::CAR_NL_GROUP_3 => PhoneNumber::CAR_NL_DEALER_3,
            self::CAR_NL_GROUP_4 => PhoneNumber::CAR_NL_DEALER_4,
            self::CAR_BE_GROUP_1 => PhoneNumber::CAR_BE_DEALER_1,
            self::CAR_BE_GROUP_2 => PhoneNumber::CAR_BE_DEALER_1,
            self::CAR_BE_GROUP_3 => PhoneNumber::CAR_BE_DEALER_1,
            self::CAR_BE_GROUP_4 => PhoneNumber::CAR_BE_DEALER_1,
            self::MARKTPLAATS_NL_GROUP_1 => PhoneNumber::MARKTPLAATS_NL,
        };
    }

    public function toSiteReference(): SiteReference
    {
        return match ($this) {
            self::CAR_NL_GROUP_1,
            self::CAR_NL_GROUP_2,
            self::CAR_NL_GROUP_3,
            self::CAR_NL_GROUP_4 => SiteReference::CAR_NL,
            self::CAR_BE_GROUP_1,
            self::CAR_BE_GROUP_2,
            self::CAR_BE_GROUP_3,
            self::CAR_BE_GROUP_4 => SiteReference::CAR_BE_NL,
            self::MARKTPLAATS_NL_GROUP_1 => SiteReference::CAR_MARKTPLAATS_NL,
        };
    }
}
