<?php

declare(strict_types=1);

namespace Dealerdirect\Generic\Enums\Motorbike;

use Dealerdirect\Generic\Enums\Interfaces\CallGroup as CallGroupInterface;
use Dealerdirect\Generic\Enums\PhoneNumber;
use Dealerdirect\Generic\Enums\PhoneNumberType;
use Dealerdirect\Generic\Enums\SiteReference;

/**
 * Contains the call group id for motorbikes from table 'dealer01_ddmoto.call_groups_desc'
 */
enum CallGroup: int implements CallGroupInterface
{
    // Motorbike NL
    case MOTOR_NL_GROUP_1 = 72;
    case MOTOR_NL_GROUP_2 = 82;
    case MOTOR_NL_GROUP_3 = 92;
    case MOTOR_NL_GROUP_4 = 102;

    // Motorbike BE
    case MOTOR_BE_GROUP_1 = 112;
    case MOTOR_BE_GROUP_2 = 122;
    case MOTOR_BE_GROUP_3 = 132;
    case MOTOR_BE_GROUP_4 = 142;

    // Motorbike DE
    case MOTOR_DE_GROUP_1 = 152;
    case MOTOR_DE_GROUP_2 = 162;

    // Scooter NL
    case SCOOTER_NL_OTHER = 4;
    case SCOOTER_NL_YOUNG_FRESH = 31;
    case SCOOTER_NL_LOW_POTENTIAL = 172;

    // Bicycle NL
    case BICYCLE_NL_ELECTRIC = 51;
    case BICYCLE_NL_NON_ELECTRIC = 61;
    case BICYCLE_NL_B2B = 71;

    public function toLabel(): string
    {
        return match ($this) {
            self::MOTOR_NL_GROUP_1,
            self::MOTOR_BE_GROUP_1,
            self::MOTOR_DE_GROUP_1 => 'Groep 1',

            self::MOTOR_NL_GROUP_2,
            self::MOTOR_BE_GROUP_2,
            self::MOTOR_DE_GROUP_2 => 'Groep 2',

            self::MOTOR_NL_GROUP_3,
            self::MOTOR_BE_GROUP_3 => 'Groep 3',

            self::MOTOR_NL_GROUP_4,
            self::MOTOR_BE_GROUP_4 => 'Groep 4',

            self::SCOOTER_NL_OTHER => 'Overig',
            self::SCOOTER_NL_YOUNG_FRESH => 'Jong & Fris',
            self::SCOOTER_NL_LOW_POTENTIAL => 'Low Potential',

            self::BICYCLE_NL_ELECTRIC => 'Elektrisch',
            self::BICYCLE_NL_NON_ELECTRIC => 'Niet Elektrisch',
            self::BICYCLE_NL_B2B => 'B2B',
        };
    }

    public function toPhoneNumber(PhoneNumberType $type): PhoneNumber
    {
        return match ($type) {
            PhoneNumberType::CONSUMER => $this->toConsumerPhoneNumber(),
            PhoneNumberType::DEALER => $this->toDealerPhoneNumber(),
            PhoneNumberType::CUSTOMER_SERVICE => $this->toCustomerServicePhoneNumber(),

            default => throw new \InvalidArgumentException("Invalid PhoneNumberType: $type->name"),
        };
    }

    public function toCustomerServicePhoneNumber(): PhoneNumber
    {
        return match ($this) {
            self::MOTOR_BE_GROUP_1,
            self::MOTOR_BE_GROUP_2,
            self::MOTOR_BE_GROUP_3,
            self::MOTOR_BE_GROUP_4 => PhoneNumber::MOTOR_NL_CUSTOMER_SERVICE, // TODO: BE customer service

            self::MOTOR_NL_GROUP_1,
            self::MOTOR_NL_GROUP_2,
            self::MOTOR_NL_GROUP_3,
            self::MOTOR_NL_GROUP_4,
            self::SCOOTER_NL_LOW_POTENTIAL,
            self::SCOOTER_NL_OTHER,
            self::SCOOTER_NL_YOUNG_FRESH => PhoneNumber::MOTOR_NL_CUSTOMER_SERVICE,

            self::MOTOR_DE_GROUP_1,
            self::MOTOR_DE_GROUP_2 => PhoneNumber::MOTOR_DE_CUSTOMER_SERVICE,

            self::BICYCLE_NL_ELECTRIC,
            self::BICYCLE_NL_NON_ELECTRIC,
            self::BICYCLE_NL_B2B => PhoneNumber::BICYCLE_NL_CUSTOMER_SERVICE,
        };
    }

    public function toConsumerPhoneNumber(): PhoneNumber
    {
        return match ($this) {
            self::MOTOR_BE_GROUP_1,
            self::MOTOR_BE_GROUP_2,
            self::MOTOR_BE_GROUP_3,
            self::MOTOR_BE_GROUP_4 => PhoneNumber::MOTOR_BE_CONSUMER_1,

            self::MOTOR_NL_GROUP_1,
            self::MOTOR_NL_GROUP_2,
            self::MOTOR_NL_GROUP_3,
            self::MOTOR_NL_GROUP_4,
            self::SCOOTER_NL_LOW_POTENTIAL,
            self::SCOOTER_NL_OTHER,
            self::SCOOTER_NL_YOUNG_FRESH => PhoneNumber::MOTOR_NL_CONSUMER_1,

            self::MOTOR_DE_GROUP_1,
            self::MOTOR_DE_GROUP_2 => PhoneNumber::MOTOR_DE_CONSUMER_1,

            self::BICYCLE_NL_ELECTRIC,
            self::BICYCLE_NL_NON_ELECTRIC,
            self::BICYCLE_NL_B2B => PhoneNumber::BICYCLE_NL_CONSUMER_1,
        };
    }

    public function toDealerPhoneNumber(): PhoneNumber
    {
        return match ($this) {
            self::MOTOR_BE_GROUP_1,
            self::MOTOR_BE_GROUP_2,
            self::MOTOR_BE_GROUP_3,
            self::MOTOR_BE_GROUP_4 => PhoneNumber::MOTOR_BE_DEALER_1,

            self::MOTOR_NL_GROUP_1,
            self::MOTOR_NL_GROUP_2,
            self::MOTOR_NL_GROUP_3,
            self::MOTOR_NL_GROUP_4,
            self::SCOOTER_NL_LOW_POTENTIAL,
            self::SCOOTER_NL_OTHER,
            self::SCOOTER_NL_YOUNG_FRESH => PhoneNumber::MOTOR_NL_DEALER_1,

            self::MOTOR_DE_GROUP_1,
            self::MOTOR_DE_GROUP_2 => PhoneNumber::MOTOR_DE_DEALER_1,

            self::BICYCLE_NL_ELECTRIC,
            self::BICYCLE_NL_NON_ELECTRIC,
            self::BICYCLE_NL_B2B => PhoneNumber::BICYCLE_NL_DEALER_1,
        };
    }

    public function toSiteReference(): SiteReference
    {
        return match ($this) {
            self::MOTOR_BE_GROUP_1,
            self::MOTOR_BE_GROUP_2,
            self::MOTOR_BE_GROUP_3,
            self::MOTOR_BE_GROUP_4 => SiteReference::MOTORBIKE_BE,

            self::MOTOR_NL_GROUP_1,
            self::MOTOR_NL_GROUP_2,
            self::MOTOR_NL_GROUP_3,
            self::MOTOR_NL_GROUP_4 => SiteReference::MOTORBIKE_NL,

            self::MOTOR_DE_GROUP_1,
            self::MOTOR_DE_GROUP_2 => SiteReference::MOTORBIKE_DE,

            self::SCOOTER_NL_OTHER,
            self::SCOOTER_NL_YOUNG_FRESH,
            self::SCOOTER_NL_LOW_POTENTIAL => SiteReference::SCOOTER_NL,

            self::BICYCLE_NL_ELECTRIC,
            self::BICYCLE_NL_NON_ELECTRIC,
            self::BICYCLE_NL_B2B => SiteReference::BICYCLE_NL,
        };
    }
}
