<?php

namespace DealerDirect\Generic\Repository;

use DealerDirect\Generic\Category\VehicleType;
use Dealerdirect\Generic\Enums;
use DealerDirect\Generic\Repository\CallGroupInformation;
use Doctrine\DBAL\Connection;
use Doctrine\DBAL\Query\QueryBuilder;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\DataProvider;
use PHPUnit\Framework\MockObject\Stub;
use PHPUnit\Framework\TestCase;

/**
 * @covers DealerDirect\Generic\Repository\CallGroupInformation
 */
#[CoversClass(CallGroupInformation::class)]
class CallGroupInformationTest extends TestCase
{
    #[DataProvider('provideValidCallGroupsWithExpectedResult')]
    public function testGetCurrentCallGroupIdShouldReturnTheClassifiedCallGroupId(
        string $vehicleType,
        int $expected,
        array|int $fetchResult
    ) {
        /** @var Connection|Stub $connection */
        $connection = $this->createStub(Connection::class);
        $queryBuilder = $this->createStub(QueryBuilder::class);
        $queryBuilder->method('select')->willReturnSelf();
        $queryBuilder->method('from')->willReturnSelf();
        $queryBuilder->method('where')->willReturnSelf();
        $queryBuilder->method('setParameter')->willReturnSelf();
        $queryBuilder->method('fetchOne')->willReturn($fetchResult['callgroup_id'] ?? $fetchResult);

        $connection->method('createQueryBuilder')->willReturn($queryBuilder);
        $connection->method('fetchAssociative')->willReturn($fetchResult);
        $connection->method('fetchOne')->willReturn($fetchResult);

        $callGroupInformation = new CallGroupInformation($connection, $vehicleType);

        $actual = $callGroupInformation->getCallGroupId(123456);

        $this->assertSame($expected, $actual);
    }

    public static function provideValidCallGroupsWithExpectedResult()
    {
        return [
            [
                VehicleType::MOTOR,
                Enums\Motorbike\CallGroup::MOTOR_NL_GROUP_1->value,
                Enums\Motorbike\CallGroup::MOTOR_NL_GROUP_1->value
            ],
            [
                VehicleType::MOTOR,
                Enums\Motorbike\CallGroup::MOTOR_DE_GROUP_2->value,
                Enums\Motorbike\CallGroup::MOTOR_DE_GROUP_2->value
            ],
            [
                VehicleType::MOTOR,
                Enums\Motorbike\CallGroup::MOTOR_BE_GROUP_4->value,
                Enums\Motorbike\CallGroup::MOTOR_BE_GROUP_4->value,
            ],
            [
                VehicleType::CAR,
                Enums\Car\CallGroup::CAR_NL_GROUP_1->value,
                Enums\Car\CallGroup::CAR_NL_GROUP_1->value
            ],
            [
                VehicleType::BICYCLE,
                Enums\Motorbike\CallGroup::BICYCLE_NL_ELECTRIC->value,
                Enums\Motorbike\CallGroup::BICYCLE_NL_ELECTRIC->value,
            ],
            [
                VehicleType::SCOOTER,
                Enums\Motorbike\CallGroup::SCOOTER_NL_YOUNG_FRESH->value,
                Enums\Motorbike\CallGroup::SCOOTER_NL_YOUNG_FRESH->value,
            ],

            // Legacy but still valid
            [VehicleType::MOTOR, 1, ['classification_id' => 1, 'reference' => 4, 'callgroup_id' => 81]],
            [VehicleType::SCOOTER, 84, ['classification_id' => null, 'reference' => 5, 'callgroup_id' => 84]],
            [VehicleType::CAR, 6, 6],
            [VehicleType::RECREATION, 51, 51],
        ];
    }
}
