<?php

namespace Dealerdirect\Generic\Storage;

use League\Flysystem\FileNotFoundException;
use League\Flysystem\Filesystem;

/**
 * Pure proxy for FilesystemInterface that allows calling FileSystem methods
 * without having to create Storage objects
 */
abstract class AbstractStorage implements FilesystemInterface
{
    ////////////////////////////// CLASS PROPERTIES \\\\\\\\\\\\\\\\\\\\\\\\\\\\

    /**
     * Two primary MIME types are important for the role of default types:
     *
     * - `text/plain` is the default value for textual files. A textual file
     *    should be human-readable and must not contain binary data.
     * - `application/octet-stream` is the default value for all other cases.
     *
     * An unknown file type should use this type. Browsers pay a particular care
     * when manipulating these files, attempting to safeguard the user to
     * prevent dangerous behaviors.
     *
     * @see https://developer.mozilla.org/en-US/docs/Web/HTTP/Basics_of_HTTP/MIME_types/Complete_list_of_MIME_types
     */
    const DEFAULT_CONTENT_TYPE = 'application/octet-stream';

    const ERROR_FILESYSTEM_MISSING = 'Filesystem is not available';

    /** @var Filesystem */
    private $filesystem;
    /** @var array */
    private static $instances = [];

    //////////////////////////// SETTERS AND GETTERS \\\\\\\\\\\\\\\\\\\\\\\\\\\

    /**
     * Returns the name of the key used in the environment to set the storage location
     *
     * @return string
     */
    abstract public static function getEnvKeyName();

    /**
     * @return Filesystem
     *
     * @throws \UnexpectedValueException
     */
    final public function getFileSystem()
    {
        if ($this->filesystem === null) {
            throw new \UnexpectedValueException(self::ERROR_FILESYSTEM_MISSING);
        }

        return $this->filesystem;
    }

    /**
     * @return static
     */
    private static function getInstance()
    {
        $class = get_called_class();

        if (array_key_exists($class, self::$instances) === false) {
            $envKey = static::getEnvKeyName();
            self::$instances[$class] = StorageFactory::getInstance($class, $envKey);
        }

        return self::$instances[$class];
    }

    //////////////////////////////// PUBLIC API \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

    /**
     * @param Filesystem|null $filesystem
     */
    final public function __construct(Filesystem $filesystem)
    {
        $this->filesystem = $filesystem;
    }

    /**
     * @param string $path
     *
     * @throws \UnexpectedValueException
     *
     * @return string
     */
    final public function getContentType($path)
    {
        $contentType = call_user_func_array([self::getInstance()->getFileSystem(), __FUNCTION__], func_get_args());

        if ($contentType === false) {
            $contentType = static::DEFAULT_CONTENT_TYPE;
        }

        return $contentType;
    }

    //////////////////////////////// PROXY API \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

    /**
     * Delete a file.
     *
     * @param string $path
     *
     * @return bool True on success, false on failure.
     *
     * @throws \UnexpectedValueException
     * @throws FileNotFoundException
     */
    final public static function delete($path)
    {
        return call_user_func_array([self::getInstance()->getFileSystem(), __FUNCTION__], func_get_args());
    }

    /**
     * Get a file's mime-type.
     *
     * @param string $path The path to the file.
     *
     * @return string|false The file mime-type or false on failure.
     *
     * @throws \UnexpectedValueException
     * @throws FileNotFoundException
     */
    final public static function getMimetype($path)
    {
        return call_user_func_array([self::getInstance()->getFileSystem(), __FUNCTION__], func_get_args());
    }

    /**
     * Check whether a file exists.
     *
     * @param string $path
     *
     * @return bool
     *
     * @throws \UnexpectedValueException
     */
    final public static function has($path)
    {
        return call_user_func_array([self::getInstance()->getFileSystem(), __FUNCTION__], func_get_args());
    }

    /**
     * Create a file or update if exists.
     *
     * @param string $path The path to the file.
     * @param string $contents The file contents.
     * @param array $config An optional configuration array.
     *
     * @return bool True on success, false on failure.
     *
     * @throws \UnexpectedValueException
     */
    final public static function put($path, $contents, array $config = [])
    {
        return call_user_func_array([self::getInstance()->getFileSystem(), __FUNCTION__], func_get_args());
    }

    /**
     * Create a file or update if exists.
     *
     * @param string $path The path to the file.
     * @param resource $resource The file handle.
     * @param array $config An optional configuration array.
     *
     * @return bool True on success, false on failure.
     *
     * @throws \UnexpectedValueException
     * @throws \InvalidArgumentException Thrown if $resource is not a resource.
     */
    final public static function putStream($path, $resource, array $config = [])
    {
        return call_user_func_array([self::getInstance()->getFileSystem(), __FUNCTION__], func_get_args());
    }

    /**
     * Read a file.
     *
     * @param string $path The path to the file.
     *
     * @return string|false The file contents or false on failure.
     *
     * @throws \UnexpectedValueException
     * @throws FileNotFoundException
     */
    final public static function read($path)
    {
        return call_user_func_array([self::getInstance()->getFileSystem(), __FUNCTION__], func_get_args());
    }

    /**
     * Retrieves a read-stream for a path.
     *
     * @param string $path The path to the file.
     *
     * @return resource|false The path resource or false on failure.
     *
     * @throws \UnexpectedValueException
     * @throws FileNotFoundException
     */
    final public static function readStream($path)
    {
        return call_user_func_array([self::getInstance()->getFileSystem(), __FUNCTION__], func_get_args());
    }
}

/*EOF*/
