<?php

namespace DealerDirect\Generic\Repository;

use Doctrine\DBAL\Connection;
use DealerDirect\Generic\Category\Site;
use DealerDirect\Generic\Category\CallGroups;
use DealerDirect\Generic\Category\VehicleType;
use DealerDirect\Generic\Category\CallGroupsMoto;
use Dealerdirect\VehicleClassification\VehicleClassificationIds;
use Dealerdirect\Generic\Enums;
use Doctrine\DBAL\ArrayParameterType;
use Psr\Log\LoggerInterface;

class CallGroupInformation
{
    // NOTE: The values come from Vehicle Classification - Vehicle model
    private const VEHICLE_TYPE_UNKNOWN = 0;
    private const VEHICLE_TYPE_CAR = 1;
    private const VEHICLE_TYPE_RECREATIONAL = 2;
    private const VEHICLE_TYPE_TRANSPORT = 3;
    private const VEHICLE_TYPE_MOTOR = 4;
    private const VEHICLE_TYPE_SCOOTER = 5;
    private const VEHICLE_TYPE_BICYCLE = 6;

    private const CUSTOMER_SERVICE_NUMBER_CAR = '+31532084446';
    private const CUSTOMER_SERVICE_NUMBER_MOTOR = '+31532084449';

    /**
     * CallGroupInformation constructor.
     *
     * @param Connection    $dbal        Database
     * @param string        $vehicleType Vehicle Type.
     */
    public function __construct(
        private Connection $dbal,
        private string $vehicleType,
        private ?LoggerInterface $logger = null
    ) {
    }

    /**
     * Returns the phone number of the call group of the given lot Id.
     *
     * @param int  $lotId                Lot Id
     * @param bool $returnConsumerNumber Consumer or Dealer
     *
     * @return bool|mixed
     */
    public function getCurrentCallGroupPhoneNumber($lotId, $returnConsumerNumber = true)
    {
        $callGroupId = $this->getCallGroupId($lotId);
        if ($callGroupId === false) {
            return false;
        }

        $vehicleType = Enums\VehicleType::tryFrom($this->vehicleType);
        $callGroup = match (true) {
            $vehicleType?->isCarBased() => Enums\Car\CallGroup::tryFrom((int) $callGroupId),
            $vehicleType?->isMotorbikeBased() => Enums\Motorbike\CallGroup::tryFrom((int) $callGroupId),
            default => null,
        };

        if ($callGroup instanceof Enums\Interfaces\CallGroup) {
            $phoneNumberType = $returnConsumerNumber ? Enums\PhoneNumberType::CONSUMER : Enums\PhoneNumberType::DEALER;

            return $callGroup->toPhoneNumber($phoneNumberType)->value;
        }

        if ($this->logger) {
            $this->logger->notice(
                'Call group not in enum, proceeding with legacy code',
                compact('callGroupId', 'lotId', 'vehicleType')
            );
        }

        $callGroupUid = $this->getCallGroupsUid($callGroupId, $returnConsumerNumber);
        if ($callGroupUid === false) {
            return false;
        }
        $groupPhoneNumber = $this->mapUidToCallGroupNumbers($callGroupUid);
        if ($groupPhoneNumber === false) {
            return false;
        }
        return $groupPhoneNumber;
    }

    /**
     * Returns the call group id of the given lot Id.
     *
     * @param int $lotId Lot Id
     *
     * @return int|false The call group id or false if not found.
     */
    public function getCallGroupId(int $lotId): int|false
    {
        $isMotor = in_array($this->vehicleType, [VehicleType::MOTOR, VehicleType::SCOOTER, VehicleType::BICYCLE], true);
        $table = $isMotor ? 'dealer01_ddmoto.call_groups_moto' : 'dealer01_ddmain.call_groups';

        $callGroupId = $this->dbal->createQueryBuilder()
            ->from($table)
            ->select('callgroup_id')
            ->where('kavelref = :id')
            ->setParameter('id', $lotId)
            ->fetchOne()
        ;

        if ($isMotor && Enums\Motorbike\CallGroup::tryFrom((int) $callGroupId) === null) {
            return $this->getCallGroupIdMotorbike($lotId);
        }

        return $callGroupId
            ? (int) $callGroupId
            : false;
    }

    /**
     * @param string $vehicleType
     * @return int[]
     */
    private function getVehicleTypeIds(): array
    {
        return match ($this->vehicleType) {
            VehicleType::CAR, VehicleType::CAMPER, VehicleType::CARAVAN, VehicleType::RECREATION => [
                self::VEHICLE_TYPE_CAR, self::VEHICLE_TYPE_TRANSPORT, self::VEHICLE_TYPE_RECREATIONAL
            ],
            VehicleType::MOTOR, VehicleType::BICYCLE, VehicleType::SCOOTER => [
                self::VEHICLE_TYPE_MOTOR, self::VEHICLE_TYPE_BICYCLE, self::VEHICLE_TYPE_SCOOTER
            ],
            default => [self::VEHICLE_TYPE_UNKNOWN],
        };
    }

    /**
     * Returns the call group for motorbikes of the given lot Id.
     *
     * @param int $lotId Lot Id
     *
     * @return int
     */
    private function getCallGroupIdMotorbike($lotId): int|false
    {
        $classificationDetails = $this->dbal->fetchAssociative(
            <<<SQL
            SELECT
                classification.classification_description_id AS classification_id,
                classification.callgroup_id,
                lots.reference
            FROM
                dealer01_ddmain.vehicle_classification AS classification
                INNER JOIN
                    dealer01_ddmoto.BO_Motorfiets AS lots
                    ON lots.id = classification.lot_id
            WHERE
                classification.lot_id = :lotId
                AND classification.vehicle_type IN(:vehicleTypes)
            LIMIT 1
            SQL,
            [
                'lotId' => $lotId,
                'vehicleTypes' => $this->getVehicleTypeIds(),
            ],
            ['vehicleTypes' => ArrayParameterType::INTEGER]
        );

        if (empty($classificationDetails)) {
            return false;
        }

        /**
         * Fallback for Lots that do not use Classification Ids (yet).
         *
         * Check if the Classification Id is available.
         * Use callgroup_id if the classification_id is not set.
         */
        if (empty($classificationDetails['classification_id'])) {
            return $classificationDetails['callgroup_id'];
        }

        return $this->mapClassificationIdToCallGroupIdForUUID(
            $classificationDetails['reference'],
            $classificationDetails['classification_id']
        );
    }

    /**
     * Maps the Classification Id's to the conventional Call Group Ids.
     */
    private function mapClassificationIdToCallGroupIdForUUID($reference, $classificationId): int
    {
        $mapping = [
            VehicleClassificationIds::CLASSIFICATION_YOUNG_ID->value => [
                Site::REF_CAR_NL => CallGroups::NL_CAR_YOUNG_FRESH_1,
                Site::REF_CAR_BE_NL => CallGroups::BE_CAR_YOUNG_FRESH,
                Site::REF_CAR_BE_FR => CallGroups::BEFR_CAR_WALLONIA,
                Site::REF_CAR_DE => CallGroups::DE_CAR_CHECKLIST,
                Site::REF_CAR_MARKTPLAATS_NL => CallGroups::NL_CAR_MP_MARKTPLAATS,
                Site::REF_SCOOTER_NL => CallGroupsMoto::NL_MOPED_YOUNG_FRESH,
                Site::REF_MOTO_NL => CallGroupsMoto::NL_MOTO_YOUNG_FRESH,
                Site::REF_MOTO_BE => CallGroupsMoto::BE_MOTO_YOUNG_FRESH,
                Site::REF_MOTO_DE => CallGroupsMoto::DE_MOTO_YOUNG_FRESH,
                Site::REF_BICYCLE_NL => CallGroupsMoto::NL_BICYCLE_ELECTRIC,
            ],
            VehicleClassificationIds::CLASSIFICATION_DEMOLITION_ID->value => [
                Site::REF_CAR_NL => CallGroups::NL_CAR_DEMOLITION,
                Site::REF_CAR_BE_NL => CallGroups::BE_CAR_YOUNG_FRESH,
                Site::REF_CAR_BE_FR => CallGroups::BEFR_CAR_WALLONIA,
                Site::REF_CAR_DE => CallGroups::DE_CAR_CHECKLIST,
                Site::REF_CAR_MARKTPLAATS_NL => CallGroups::NL_CAR_MP_MARKTPLAATS,
                Site::REF_SCOOTER_NL => CallGroupsMoto::NL_MOPED_DEMOLITION,
                Site::REF_MOTO_NL => CallGroupsMoto::NL_MOTO_DEMOLOTION,
                Site::REF_MOTO_BE => CallGroupsMoto::BE_MOTO_DEMOLITION,
                Site::REF_MOTO_DE => CallGroupsMoto::DE_MOTO_DEMOLITION,
                Site::REF_BICYCLE_NL => CallGroupsMoto::NL_BICYCLE_ELECTRIC,
            ],
            VehicleClassificationIds::CLASSIFICATION_RECREATIONAL_ID->value => [
                Site::REF_CAR_NL => CallGroups::NL_CAR_RECREATION,
                Site::REF_CAR_BE_NL => CallGroups::BE_CAR_RECREATION,
                Site::REF_CAR_BE_FR => CallGroups::BEFR_CAR_WALLONIA,
                Site::REF_CAR_DE => CallGroups::DE_CAR_CHECKLIST,
                Site::REF_CAR_MARKTPLAATS_NL => CallGroups::NL_CAR_MP_MARKTPLAATS,
                Site::REF_SCOOTER_NL => CallGroupsMoto::NL_MOPED_YOUNG_FRESH,
                Site::REF_MOTO_NL => CallGroupsMoto::NL_MOTO_YOUNG_FRESH,
                Site::REF_MOTO_BE => CallGroupsMoto::BE_MOTO_YOUNG_FRESH,
                Site::REF_MOTO_DE => CallGroupsMoto::DE_MOTO_YOUNG_FRESH,
                Site::REF_BICYCLE_NL => CallGroupsMoto::NL_BICYCLE_ELECTRIC,
            ],
            VehicleClassificationIds::CLASSIFICATION_TRANSPORT_ID->value => [
                Site::REF_CAR_NL => CallGroups::NL_CAR_ORDER_FREIGHT,
                Site::REF_CAR_BE_NL => CallGroups::BE_CAR_ORDER_FREIGHT,
                Site::REF_CAR_BE_FR => CallGroups::BEFR_CAR_WALLONIA,
                Site::REF_CAR_DE => CallGroups::DE_CAR_CHECKLIST,
                Site::REF_CAR_MARKTPLAATS_NL => CallGroups::NL_CAR_MP_MARKTPLAATS,
                Site::REF_SCOOTER_NL => CallGroupsMoto::NL_MOPED_YOUNG_FRESH,
                Site::REF_MOTO_NL => CallGroupsMoto::NL_MOTO_YOUNG_FRESH,
                Site::REF_MOTO_BE => CallGroupsMoto::BE_MOTO_YOUNG_FRESH,
                Site::REF_MOTO_DE => CallGroupsMoto::DE_MOTO_YOUNG_FRESH,
                Site::REF_BICYCLE_NL => CallGroupsMoto::NL_BICYCLE_ELECTRIC,
            ],
            VehicleClassificationIds::CLASSIFICATION_BUDGET_ID->value => [
                Site::REF_CAR_NL => CallGroups::NL_CAR_BUDGET_1,
                Site::REF_CAR_BE_NL => CallGroups::BE_CAR_BUDGET,
                Site::REF_CAR_BE_FR => CallGroups::BEFR_CAR_WALLONIA,
                Site::REF_CAR_DE => CallGroups::DE_CAR_CHECKLIST,
                Site::REF_CAR_MARKTPLAATS_NL => CallGroups::NL_CAR_MP_MARKTPLAATS,
                Site::REF_SCOOTER_NL => CallGroupsMoto::NL_MOPED_YOUNG_FRESH,
                Site::REF_MOTO_NL => CallGroupsMoto::NL_MOTO_YOUNG_FRESH,
                Site::REF_MOTO_BE => CallGroupsMoto::BE_MOTO_YOUNG_FRESH,
                Site::REF_MOTO_DE => CallGroupsMoto::DE_MOTO_YOUNG_FRESH,
                Site::REF_BICYCLE_NL => CallGroupsMoto::NL_BICYCLE_ELECTRIC,
            ],
            VehicleClassificationIds::CLASSIFICATION_EXPORT_ID->value => [
                Site::REF_CAR_NL => CallGroups::NL_CAR_EXPORT_1,
                Site::REF_CAR_BE_NL => CallGroups::BE_CAR_EXPORT,
                Site::REF_CAR_BE_FR => CallGroups::BEFR_CAR_WALLONIA,
                Site::REF_CAR_DE => CallGroups::DE_CAR_CHECKLIST,
                Site::REF_CAR_MARKTPLAATS_NL => CallGroups::NL_CAR_MP_MARKTPLAATS,
                Site::REF_SCOOTER_NL => CallGroupsMoto::NL_MOPED_YOUNG_FRESH,
                Site::REF_MOTO_NL => CallGroupsMoto::NL_MOTO_YOUNG_FRESH,
                Site::REF_MOTO_BE => CallGroupsMoto::BE_MOTO_YOUNG_FRESH,
                Site::REF_MOTO_DE => CallGroupsMoto::DE_MOTO_YOUNG_FRESH,
                Site::REF_BICYCLE_NL => CallGroupsMoto::NL_BICYCLE_ELECTRIC,
            ],
            VehicleClassificationIds::CLASSIFICATION_OTHER_ID->value => [
                Site::REF_CAR_NL => CallGroups::NL_CAR_OTHER_1,
                Site::REF_CAR_BE_NL => CallGroups::BE_CAR_OTHER,
                Site::REF_CAR_BE_FR => CallGroups::BEFR_CAR_WALLONIA,
                Site::REF_CAR_DE => CallGroups::DE_CAR_CHECKLIST,
                Site::REF_CAR_MARKTPLAATS_NL => CallGroups::NL_CAR_MP_MARKTPLAATS,
                Site::REF_SCOOTER_NL => CallGroupsMoto::NL_MOPED_YOUNG_FRESH,
                Site::REF_MOTO_NL => CallGroupsMoto::NL_MOTO_YOUNG_FRESH,
                Site::REF_MOTO_BE => CallGroupsMoto::BE_MOTO_YOUNG_FRESH,
                Site::REF_MOTO_DE => CallGroupsMoto::DE_MOTO_YOUNG_FRESH,
                Site::REF_BICYCLE_NL => CallGroupsMoto::NL_BICYCLE_ELECTRIC,
            ],
            VehicleClassificationIds::CLASSIFICATION_PREMIUM_ID->value => [
                Site::REF_CAR_NL => CallGroups::NL_CAR_PREMIUM_CLASSIC,
                Site::REF_CAR_BE_NL => CallGroups::BE_CAR_PREMIUM_CLASSIC,
                Site::REF_CAR_BE_FR => CallGroups::BEFR_CAR_WALLONIA,
                Site::REF_CAR_DE => CallGroups::DE_CAR_CHECKLIST,
                Site::REF_CAR_MARKTPLAATS_NL => CallGroups::NL_CAR_MP_MARKTPLAATS,
                Site::REF_SCOOTER_NL => CallGroupsMoto::NL_MOPED_YOUNG_FRESH,
                Site::REF_MOTO_NL => CallGroupsMoto::NL_MOTO_YOUNG_FRESH,
                Site::REF_MOTO_BE => CallGroupsMoto::BE_MOTO_YOUNG_FRESH,
                Site::REF_MOTO_DE => CallGroupsMoto::DE_MOTO_YOUNG_FRESH,
                Site::REF_BICYCLE_NL => CallGroupsMoto::NL_BICYCLE_ELECTRIC,
            ],
        ];

        return $mapping[$classificationId][$reference] ?? 0;
    }

    /**
     * Returns the Pure Cloud Uid of the given call group.
     *
     * @param int  $callGroupId          Call Group Id
     * @param bool $returnConsumerNumber Consumer or Dealer
     *
     * @return bool
     */
    private function getCallGroupsUid($callGroupId, $returnConsumerNumber)
    {
        return in_array($this->vehicleType, [VehicleType::MOTOR, VehicleType::SCOOTER, VehicleType::BICYCLE])
            ? $this->getCallGroupsUidMotorbike($callGroupId, $returnConsumerNumber)
            : $this->getCallGroupsUidCar($callGroupId, $returnConsumerNumber);
    }

    /**
     * Returns the Pure Cloud Uid for cars of the given call group.
     *
     * @param int  $callGroupId          Call Group Id
     * @param bool $returnConsumerNumber Consumer or Dealer
     *
     * @return bool
     */
    private function getCallGroupsUidCar($callGroupId, $returnConsumerNumber)
    {
        $column = $returnConsumerNumber ? 'pc_consumer_uid' : 'pc_dealer_uid';
        $results = $this->dbal->fetchAssociative(
            <<<SQL
            SELECT
                call_groups_uids.pc_consumer_uid,
                call_groups_uids.pc_dealer_uid
            FROM
                dealer01_ddmain.call_groups_uids
            WHERE
                call_groups_uids.callgroup_id = :callGroupId
            LIMIT 1
            SQL,
            compact('callGroupId')
        );

        return $results[$column] ?? false;
    }

    /**
     * Returns the Pure Cloud Uid for motorbikes of the given call group.
     *
     * @param int  $callGroupId          Call Group Id
     * @param bool $returnConsumerNumber Consumer or Dealer
     *
     * @return bool
     */
    private function getCallGroupsUidMotorbike($callGroupId, $returnConsumerNumber)
    {
        $column = $returnConsumerNumber ? 'pc_consumer_uid' : 'pc_dealer_uid';
        $results = $this->dbal->fetchAssociative(
            <<<SQL
            SELECT
                call_groups_moto_uids.pc_consumer_uid,
                call_groups_moto_uids.pc_dealer_uid
            FROM
                dealer01_ddmoto.call_groups_moto_uids
            WHERE
                call_groups_moto_uids.callgroup_id = :callGroupId
            LIMIT 1
            SQL,
            compact('callGroupId')
        );

        return $results[$column] ?? false;
    }

    /**
     * Returns the known phone number for the given Uid.
     *
     * @param string $uid Uid
     *
     * @return bool|string
     */
    public function mapUidToCallGroupNumbers($uid)
    {
        // phpcs:disable Generic.Files.LineLength
        $phoneNumber = match ($uid) {
            // Car NL
            '8492d2c3-bfbe-4171-8b7e-1761551325d2' => Enums\PhoneNumber::CAR_NL_CUSTOMER_SERVICE, // CAR NL customer service
            '4331cc30-788b-445d-802d-b7b20b15ea1e' => Enums\PhoneNumber::CAR_NL_CONSUMER_1, // Groep 1 consumers
            '2fcccd52-c15d-47a4-baeb-f1fde47e1f92' => Enums\PhoneNumber::CAR_NL_DEALER_1, // Groep 1 dealers
            'fa1b642e-b100-452a-9aea-870466c6ea2c' => Enums\PhoneNumber::CAR_NL_CONSUMER_2, // Groep 2 consumers
            '78cfb885-243d-4a50-a4eb-623d16e28907' => Enums\PhoneNumber::CAR_NL_DEALER_2, // Groep 2 dealers
            'd4a46e1e-cd16-419c-9090-7d7f3df5c063' => Enums\PhoneNumber::CAR_NL_CONSUMER_3, // Groep 3 consumers
            '6ab85949-836c-4a88-b241-d3fced4bd87e' => Enums\PhoneNumber::CAR_NL_DEALER_3, // Groep 3 dealers
            'd65fe590-3a12-4150-8e4e-caa96b290dd4' => Enums\PhoneNumber::CAR_NL_CONSUMER_4, // Groep 4 consumers
            'b2fc3adb-a573-4c58-9869-503289e697a2' => Enums\PhoneNumber::CAR_NL_DEALER_4, // Groep 4 dealers

            // Car BE
            '7e7ce2a4-e5e6-4b2b-a294-d5672b6b778c' => Enums\PhoneNumber::CAR_NL_CUSTOMER_SERVICE, // CAR BE customer service
            '22d20103-85d1-4b07-b858-63d1dbb09819' => Enums\PhoneNumber::CAR_BE_CONSUMER_1, // Groep 1 consumers
            '89a78b9b-f02d-4d8e-9953-7f2172bb854f' => Enums\PhoneNumber::CAR_BE_DEALER_1, // Groep 1 dealers

            // Marktplaats NL
            '2e1656e0-1c44-47cb-be4e-2f2fc6140e6a' => Enums\PhoneNumber::MARKTPLAATS_NL, // 1 number for all

            // Motor NL
            '9d4f9e27-e4d1-44c0-b3b9-e24be8babbb4' => Enums\PhoneNumber::MOTOR_NL_CUSTOMER_SERVICE, // Motor NL Groep 1-4 customer service
            '05ddec61-da33-419f-b9f9-407dd210c749' => Enums\PhoneNumber::MOTOR_NL_CONSUMER_1, // Motor NL Groep 1-4 consumers
            'd87d3eb5-f829-4508-9d6f-05f0cf1dd6fd' => Enums\PhoneNumber::MOTOR_NL_DEALER_1, // Motor NL Groep 1-4 dealers

            // Scooter NL
            '05f8d1fd-ae34-4d5f-a39a-78ac973399bd' => Enums\PhoneNumber::MOTOR_NL_CUSTOMER_SERVICE, // Scooter NL customer service
            'ad7d05f4-98c4-421f-b08a-6983fd2a8eef' => Enums\PhoneNumber::MOTOR_NL_CONSUMER_1, // Scooter NL consumers
            'c129212f-46a5-40c1-a7a8-eeab5e6e9e93' => Enums\PhoneNumber::MOTOR_NL_DEALER_1, // Scooter NL dealers

            // Bicycle NL
            '6b1d49bb-2da4-4163-85d9-7a9a6aab02b6' => Enums\PhoneNumber::BICYCLE_NL_CUSTOMER_SERVICE, // Bicycle NL customer service
            'f8b3d642-26d3-4ba9-9e3c-c1eede7d47a1' => Enums\PhoneNumber::BICYCLE_NL_CONSUMER_1, // Bicycle NL consumers
            'cdede338-fbcb-4e25-b6c5-8b98fc6200ca' => Enums\PhoneNumber::BICYCLE_NL_DEALER_1, // Bicycle NL dealers

            // Motor BE
            // TODO: Check this, as the old situation points to the same customer service number as Motor NL
            'e63c8f20-1e1e-4375-86f7-4b716f02c4b1' => Enums\PhoneNumber::MOTOR_NL_CUSTOMER_SERVICE, // Motor BE customer service
            '92f1b005-55ef-462b-9723-5506df485d8b' => Enums\PhoneNumber::MOTOR_BE_CONSUMER_1, // Motor BE consumers
            '328d3be9-11e3-4000-be0a-3f139dfde0a7' => Enums\PhoneNumber::MOTOR_BE_DEALER_1, // Motor BE dealers

            // Motor DE
            // TODO: Check this, as the old situation points to the same customer service number as Motor NL
            '48f907b3-ba78-4e6c-934b-bc9e08d18fd6' => Enums\PhoneNumber::MOTOR_NL_CUSTOMER_SERVICE, // Motor DE customer service
            '91e2b136-e60c-4092-85bb-882266c486e8' => Enums\PhoneNumber::MOTOR_DE_CONSUMER_1, // Motor DE consumers
            '9adab05c-0e64-4120-8f0a-890653bd1e9c' => Enums\PhoneNumber::MOTOR_DE_DEALER_1, // Motor DE dealers


            // Legacy, these should all be inactive..
            '00221f20-4b52-4971-87c5-309ced6925df' => '+31537600081', // NL Car Bestel en Vracht Consumers,
            '0cc8a687-5028-419a-af30-648e4598ede8' => '+31537440542', // NL Car Bestel en Vracht Dealers.
            '0068aafd-a811-4a06-9fe8-7a3526612632' => '+31532084216', // NL Car Overig Laag 2 Consumers.
            'aaf17097-5c05-4dad-b020-c896868b6403' => '+31537600207', // NL Car Overig Laag 2 Dealers.
            '0e1b3771-e54b-4ccb-a9ed-486ef7427a15' => '+31532084222', // NL Car Quality Control.
            '225eac22-877d-464d-a884-2c082c011aef' => '+31537600082', // NL Car Recreatie Consumers.
            'dc579b00-4316-4e2a-bd63-ab4952ffd0e9' => '+31537440544', // NL Car Recreatie Dealers.
            '2d4219b5-7220-4280-ad63-fe3c358e00bf' => '+31537600204', // NL Car Jong en Fris 1 Consumers
            '2d9cba52-f45b-42fa-9052-9149ffbb641e' => '+31537440539', // NL Car Jong en Fris 1 Dealers.
            '3955a12b-5b9f-4134-89af-8f3a2782bf5b' => '+31532084215', // NL Car Overig Laag 1 Consumers.
            'aa3e02c0-894d-4ca2-8a2d-267b30c67cc4' => '+31537440541', // NL Car Overig Laag 1 Dealers.
            '4ddcef0b-6c6f-471a-a9c5-7ec00d7deb82' => '+31532084220', // NL Car Export Consumers.
            'e31290a4-8562-4b2f-bb36-ac1093593712' => '+31537440538', // NL Car Export Dealers.
            '62a3908e-c900-42a0-9b06-d24e6080da81' => '+31532084217', // NL Car Overig Laag 3 Consumers.
            '65956e61-48c5-483a-9e01-dadaef213c24' => '+31532084223', // NL Car Overig Laag 3 Dealers.
            '82aef871-2ee9-4b7b-b083-a580659afa84' => '+31537600205', // NL Car Schade en Sloop Consumers.
            '029903f8-8764-4b69-9176-98e8374aea2d' => '+31537440543', // NL Car Schade en Sloop Dealers.
            '909119c7-5b59-4646-8578-de6b7a257c18' => '+31532084218', // NL Car Overig Hoog 1 Consumers.
            'b68c1805-6709-4cb4-9dec-4e1c9e91a49e' => '+31537440540', // NL Car Overig Hoog 1 Dealers.
            '956289d7-1287-47cd-92c3-1fa3e6170e44' => '+31532084221', // NL Car Export 2 Consumers.
            '0fb79d6d-d5c0-41c4-b353-78cba38298dc' => '+31537600206', // NL Car Export 2 Dealers.
            'a84c097c-cd12-4505-971f-02a87d8f69c8' => '+31532084219', // NL Car Overig Hoog 2 Consumers.
            'db400ded-c4dd-4d89-a3d5-26a4d55e3800' => '+31537600210', // NL Car Overig Hoog 2 Dealers.
            'd57276fc-1564-470e-94e3-06246477a50f' => '+31537600942', // NL Car Jong en Fris 2 Consumers.
            '56ffcf53-5759-4185-ae1a-c89fb2ea6b94' => '+31537600943', // NL Car Jong en Fris 2 Dealers.
            'd9035c91-5df9-4e7c-981a-75bac6e74aca' => '', // DE Car Consumers.
            '9ae4250f-e558-4295-bb89-afb4a0659093' => '+4925629993011', // DE Car Dealers.
            'fe76429d-a88b-4ab1-ad59-b675a4f80d3c' => '+3216980389', // BE Car Consumers.
            '7c6424a8-180d-48a2-b4b6-10f046329422' => '+3216980381', // BE Car Dealers.
            // 'ad7d05f4-98c4-421f-b08a-6983fd2a8eef' => '+31537600030', // NL Motorbike.
            // 'c129212f-46a5-40c1-a7a8-eeab5e6e9e93' => '+31537600029', // NL Motorbike Dealers.
            '7160ef68-0a5f-4034-99fd-f8d1bebed276' => '+3216980331', // BE Motorbike.
            'b7302add-ccdf-4a99-8b63-04a36fe6616d' => '+3216980341', // BE Motorbike Dealers.
            'ca2551ea-0fc8-4434-8481-b52a07fb63a5' => '+4925629993010', // DE Motorbike.
            '79d9f2c8-6c46-40fb-bc6d-4a953c1b22fa' => '+4925629993011', // DE Motorbike Dealers.
            // NL Car Jong En Fris 1 Customer Service
            '2ede71da-f329-4472-8b85-f2b2494cb55d' => self::CUSTOMER_SERVICE_NUMBER_CAR,
            // Schade en sloop Customer-care
            '124202d7-f162-4e43-b32e-d9bb5244fa1d' => self::CUSTOMER_SERVICE_NUMBER_CAR,
            '7151683d-ce08-4a83-889d-a455222018ca' => self::CUSTOMER_SERVICE_NUMBER_CAR, // Export Customer-care
            '1dcb109f-7e95-44a1-82ee-8ea28d9d51a3' => self::CUSTOMER_SERVICE_NUMBER_CAR, // Overig Hoog Customer-care
            '06b024d6-93e2-4fc6-a8f0-341bbaeab82b' => '+3216980382', // BE Car CustomerService
            '943a99f8-10e8-4d4d-aa13-0b6856bcd389' => '+4925629993012', // DE Moto Kundenservice Dealerdirect
            '10162f22-84e8-4f66-8202-8704c8390556' => self::CUSTOMER_SERVICE_NUMBER_CAR, // NL car Customer-care
            // NL moto IkWilVanMijnMotorFietsAf Customer-care
            // '05f8d1fd-ae34-4d5f-a39a-78ac973399bd' => self::CUSTOMER_SERVICE_NUMBER_MOTOR,
            // NL CAR OverigLaag1 Customer-care
            'b2e5630a-800e-4429-84aa-c8467a92ccae' => self::CUSTOMER_SERVICE_NUMBER_CAR,
            // 'f8b3d642-26d3-4ba9-9e3c-c1eede7d47a1' => '+31532084444', // NL Fietsen Consumer
            // 'cdede338-fbcb-4e25-b6c5-8b98fc6200ca' => '+31532084442', // NL Fietsen Dealer
            // '6b1d49bb-2da4-4163-85d9-7a9a6aab02b6' => '+31532084443', // NL Fietsen customercare
            '2b449002-6c5c-4ca5-9406-f95806d97821' => '+31857733443', // NL Marktplaats

            default => false,
        };

        // phpcs:enable Generic.Files.LineLength

        if (empty($phoneNumber)) {
            return false;
        }

        if (! $phoneNumber instanceof Enums\PhoneNumber) {
            $enum = Enums\PhoneNumber::tryFrom($phoneNumber);

            if ($this->logger) {
                $this->logger->notice('Phone number from legacy uid mapping', [
                    'uid' => $uid,
                    'phoneNumber' => $phoneNumber,
                    'enum' => $enum,
                ]);
            }

            $phoneNumber = $enum ?? $phoneNumber;
        }

        return $phoneNumber instanceof Enums\PhoneNumber
            ? $phoneNumber->value
            : $phoneNumber;
    }

    /**
     * Returns the known phone number for the given Uid.
     * TODO: get this info from the to be build phoneNumber information class
     *
     * @param string $uid Uid
     *
     * @return bool|mixed
     */
    public function mapNumberToBrand($phoneNumber)
    {
        $enum = Enums\PhoneNumber::tryFrom($phoneNumber);
        if (!is_null($enum)) {
            return $enum->toBrand();
        }

        if ($this->logger) {
            $this->logger->notice(
                'Phone number not in Enums\PhoneNumber, proceed with legacy mapping',
                compact('phoneNumber'),
            );
        }

        // Legacy mapping
        return match ($phoneNumber) {
            '+31537600081' => 'Ikwilvanmijnautoaf', // NL Car Bestel en Vracht Consumers,
            '+31537440542' => 'Dealerdirect', // NL Car Bestel en Vracht Dealers.
            '+31532084216' => 'Ikwilvanmijnautoaf', // NL Car Overig Laag 2 Consumers.
            '+31537600207' => 'Dealerdirect', // NL Car Overig Laag 2 Dealers.
            '+31532084222' => 'Ikwilvanmijnautoaf', // NL Car Quality Control.
            '+31537600082' => 'Ikwilvanmijnautoaf', // NL Car Recreatie Dealers.
            '+31537440544' => 'Dealerdirect', // NL Car Recreatie Consumers.
            '+31537600204' => 'Ikwilvanmijnautoaf', // NL Car Jong en Fris 1 Consumers
            '+31537440539' => 'Dealerdirect', // NL Car Jong en Fris 1 Dealers.
            '+31532084215' => 'Ikwilvanmijnautoaf', // NL Car Overig Laag 1 Consumers.
            '+31537440541' => 'Dealerdirect', // NL Car Overig Laag 1 Dealers.
            '+31532084220' => 'Ikwilvanmijnautoaf', // NL Car Export Consumers.
            '+31537440538' => 'Dealerdirect', // NL Car Export Dealers.
            '+31532084217' => 'Ikwilvanmijnautoaf', // NL Car Overig Laag 3 Consumers.
            '+31532084223' => 'Dealerdirect', // NL Car Overig Laag 3 Dealers.
            '+31537600205' => 'Ikwilvanmijnautoaf', // NL Car Schade en Sloop Consumers.
            '+31537440543' => 'Dealerdirect', // NL Car Schade en Sloop Dealers.
            '+31532084218' => 'Ikwilvanmijnautoaf', // NL Car Overig Hoog 1 Consumers.
            '+31537440540' => 'Dealerdirect', // NL Car Overig Hoog 1 Dealers.
            '+31532084221' => 'Ikwilvanmijnautoaf', // NL Car Export 2 Consumers.
            '+31537600206' => 'Dealerdirect', // NL Car Export 2 Dealers.
            '+31532084219' => 'Ikwilvanmijnautoaf', // NL Car Overig Hoog 2 Consumers.
            '+31537600210' => 'Dealerdirect', // NL Car Overig Hoog 2 Dealers.
            '+31537600942' => 'Ikwilvanmijnautoaf', // NL Car Jong en Fris 2 Consumers.
            '+31537600943' => 'Dealerdirect', // NL Car Jong en Fris 2 Dealers.
            '' => '', // // DE Car Consumers.
            '+4925629993011' => 'Dealerdirect', // DE Car Dealers.
            '+3216980389' => 'Ikwilvanmijnautoaf', // BE Car Consumers.
            '+3216980381' => 'Dealerdirect', // BE Car Dealers.
            '+31537600030' => 'Ikwilvanmijnmotoraf', // NL Motorbike.
            '+31537600029' => 'MotorDealerdirect', // NL Motorbike Dealers.
            '+3216980331' => 'Ikwilvanmijnmotoraf', // BE Motorbike.
            '+3216980341' => 'MotorDealerdirect', // BE Motorbike Dealers.
            '+4925629993010' => 'ichwillmeinmotorradloswerden', // DE Motorbike.
            '+4925629993011' => 'MotorDealerdirect', // DE Motorbike Dealers.
            self::CUSTOMER_SERVICE_NUMBER_CAR => 'Ikwilvanmijnautoaf', // NL Car Jong En Fris 1 Customer Service
            '+31532084428' => 'Ikwilvanmijnautoaf', // Schade en sloop Customer-care
            '+31532084418' => 'Ikwilvanmijnautoaf', // Export Customer-care
            '+31532084437' => 'Ikwilvanmijnautoaf', // Overig Hoog Customer-care
            '+3216980382' => 'Ikwilvanmijnautoaf', // BE Car CustomerService
            '+4925629993012' => 'Dealerdirect', // DE Moto Kundenservice Dealerdirect
            '+31537600027' => 'Ikwilvanmijnautoaf', // NL car Customer-care
            self::CUSTOMER_SERVICE_NUMBER_MOTOR => 'Ikwilvanmijnmotoraf', // NL moto Ikwilvanmijnmotoraf Customer-care
            '+31532084431' => 'Ikwilvanmijnautoaf', // NL CAR OverigLaag1 Customer-care
            '+31532084444' => 'Ikwilvanmijnfietsaf', // NL fiets consumers
            '+31532084442' => 'Dealerdirect', // NL fiets dealers
            '+31532084443' => 'Ikwilvanmijnfietsaf', // NL fiets customer care
            '+31857733443' => 'Marktplaats',

            default => 'Dealerdirect',
        };
    }
}
