<?php

namespace DealerDirect\Generic\Repository;

use DealerDirect\Generic\Category\CountryTerritoryCode;
use DealerDirect\Generic\Category\Locale as LocaleCategory;
use DealerDirect\Generic\Category\Territory;
use DealerDirect\Generic\Category\Language;

/**
 * Immutable object holding the value of an RFC-5646 compliant "Tag for Identifying Languages"
 *
 * Also known as a "Locale" the form of such a tag is:
 *
 *     language[_territory][.codeset][@modifier]
 *
 * Where:
 * - language is a two-letter, lowercase and ISO_639-1 compliant
 * - territory is two-letter, uppercase and ISO 3166-1 alpha-2 compliant
 *
 * Despite the RFC dictating values as *case insensitive*, casing conventions are
 * upheld for the sake of being recognisable as values used elsewhere.
 */
class Locale implements ImmutableInterface
{
    ////////////////////////////// CLASS PROPERTIES \\\\\\\\\\\\\\\\\\\\\\\\\\\\
    /** @var string */
    private $language;
    /** @var string */
    private $territory;

    //////////////////////////// SETTERS AND GETTERS \\\\\\\\\\\\\\\\\\\\\\\\\\\
    /** @return string */
    final public function getLanguage()
    {
        return $this->language;
    }

    /** @return string */
    final public function getTerritory()
    {
        return $this->territory;
    }

    //////////////////////////////// PUBLIC API \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\
    /**
     * Locale constructor.
     *
     * @param string $language
     * @param string $territory
     *
     * @throws \InvalidArgumentException
     */
    final public function __construct($language, $territory = '')
    {
        $language = mb_strtolower($language);
        $localeId = $language;

        if ($territory !== '') {
            $territory = mb_strtoupper($territory);
            $localeId = $localeId . '_' . $territory;
        }
        LocaleCategory::assertExists($localeId);

        $this->language = $language;
        $this->territory = $territory;
    }

    /**
     * Creates a Locale instance from a locale string.
     *
     * @param $localeStr The locale in string format
     *
     * @return Locale The Locale instance
     */
    final public static function fromString($localeStr)
    {
        $localeParts = explode('_', $localeStr);
        if (count($localeParts) !== 2) {
            throw new \InvalidArgumentException('Invalid locale string');
        }

        return new self(strtolower($localeParts[0]), strtoupper($localeParts[1]));
    }


    /** @return string */
    final public function __toString()
    {
        $locale = $this->getLanguage();

        if ($this->getTerritory() !== '') {
            $locale .= '_' . $this->getTerritory();
        }

        return $locale;
    }
}
